<?php declare(strict_types = 1);

namespace Currency\AdminModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Currency\AdminModule\Model\Currencies;
use Currency\AdminModule\Model\Security;
use Currency\Model\Config;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use h4kuna\Exchange\Driver\Cnb;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\Html;

class CurrenciesGrid extends BaseControl
{
	/** @var Currencies */
	protected $currenciesService;

	/** @var Security */
	protected $security;

	/** @var Config */
	protected $config;

	public function __construct(Currencies $currencies, Config $config, Security $security)
	{
		$this->currenciesService = $currencies;
		$this->config            = $config;
		$this->security          = $security;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleDelete($id): void
	{
		if (!$this->security->delete())
			throw new ForbiddenRequestException();

		$presenter = $this->getPresenter();

		if ($this->currenciesService->remove($id))
			$presenter->flashMessageSuccess('default.deleted');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleSync($id): void
	{
		$presenter = $this->getPresenter();
		$currency  = $this->currenciesService->get($id);
		$list      = (new Cnb\Day())->download(null, $this->config->get('whitelist'));

		if (!$currency || !$list->offsetGet($currency->getCode())) {
			$presenter->flashMessageDanger('currency.forbiddenCurrency');
		} else {
			$this->em->beginTransaction();
			try {
				$defaultCurrency = $list->offsetGet($this->config->get('default'));
				$history = new CurrencyHistory($currency);
				$this->em->persist($history);
				$currency->rate = Strings::formatEntityDecimal($list->offsetGet($currency->getCode())->rate / $defaultCurrency->rate);
				$this->em->persist($currency);
				$this->em->flush();
				$this->em->commit();
				$this['grid']->reload();
				$presenter->flashMessageSuccess('currency.updated');
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();

				$presenter->flashMessageDanger('currency.updateError');
			}
		}

		$presenter->redrawControl('flashes');
	}

	public function handleHistory($id): void
	{
		$history = $this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
			->select('c.code, ch.home, ch.rate, ch.created')
			->innerJoin('ch.currency', 'c')
			->where('ch.currency = :currency')->setParameter('currency', $id)
			->orderBy('ch.created', 'DESC')
			->getQuery()->getArrayResult();

		$this->template->history    = $history;
		$this->template->modalTitle = $this->t('currency.title.currencyHistory', ['currency' => $history['code']]);
		$this->template->modal      = 'history';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		BaseDataGrid::$iconPrefix = '';

		$qb = $this->currenciesService->getEr()->createQueryBuilder('c')
			->addSelect('country, site')
			->leftJoin('c.country', 'country')
			->leftJoin('c.site', 'site')
			->orderBy('site.ident')->addOrderBy('c.code');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('default', 'currency.currency.default')->setRenderer(function(Currency $row) {
			return $this->config->get('default') == $row->getCode()
				? Html::el('span', ['class' => 'btn btn-xs btn-success'])->addHtml(Html::el('i', ['class' => 'fa fa-check']))
				: '';
		})->setAlign('center');
		$grid->addColumnText('site', 'currency.currency.eshop', 'site.ident');
		$grid->addColumnText('code', 'currency.currency.code');
		$grid->addColumnText('symbol', 'currency.currency.symbol')->setAlign('center');
		$grid->addColumnText('title', 'currency.currency.title');
		$grid->addColumnText('nativeTitle', 'currency.currency.nativeTitle');
		$grid->addColumnText('country', 'currency.currency.country', 'country.name');
		$grid->addColumnNumber('rate', 'currency.currency.rate')->setFormat(3, ',', ' ');
		$grid->addColumnStatus('isActive', 'currency.currency.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('sync', '', 'sync!')->setIcon('fas fa-cloud-download-alt')->setBsType('success')
			->setRenderCondition(function(Currency $row) { return (bool)$row->sync; })->addClass('ajax');
		$grid->addAction('history', '', 'history!')->setIcon('fas fa-history')->setBsType('secondary')
			->addClass('ajax');
		if ($this->security->edit())
			$grid->addAction('edit', '', ':editCurrency!')->setIcon('fas fa-pencil-alt')->setBsType('primary')
				->addClass('ajax');
		if ($this->security->delete())
			$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
				->setConfirm('default.reallyDelete');

		// Prototype
		$grid->getColumn('default')->getElementPrototype('th')->class[]  = 'w1nw';
		$grid->getColumn('symbol')->getElementPrototype('th')->class[]   = 'w1nw';
		$grid->getColumn('symbol')->getElementPrototype('td')->class[]   = 'w1nw';
		$grid->getColumn('code')->getElementPrototype('td')->class[]     = 'w1nw';
		$grid->getColumn('isActive')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->currenciesService->setActive($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}
}
