<?php declare(strict_types = 1);

namespace Currency\Model\Driver;

use Currency\Model\Config;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use h4kuna\Exchange\Driver\Cnb\Day;
use h4kuna\Exchange\Driver\Cnb\Property;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Utils\DateTime;

class Database extends Day
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Currencies */
	protected $currenciesService;

	/** @var array */
	protected $cFromDB;

	public function __construct(EntityManagerDecorator $em, Currencies $currencies)
	{
		$this->em                = $em;
		$this->currenciesService = $currencies;
	}

	protected function getEmptyData(): array
	{
		$data = [];
		foreach (Config::load('whitelist') as $v)
			$data[$v] = [];

		$data[Config::load('default')] = [
			'code' => Config::load('default'),
			'home' => 1,
			'rate' => 1,
		];

		return $data;
	}

	/**
	 * @param \DateTimeInterface|null $date
	 *
	 * @return iterable
	 * @throws \Exception
	 */
	protected function loadFromSource(?\DateTimeInterface $date): iterable
	{
		$this->setDate('Y-m-d', ($date ?: new DateTime())->format('Y-m-d'));
		$data = $this->getEmptyData();

		if ($date) {
			foreach ($this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
				         ->select('c.code as code, ch.home, ch.rate, ch.created, c.id as id')
				         ->innerJoin('ch.currency', 'c')
				         ->where('ch.created >= :from')->andWhere('ch.created <= :to')
				         ->setParameters([
					         'from' => (clone $date)->setTime(0, 0, 0),
					         'to'   => (clone $date)->setTime(23, 59, 59),
				         ])
				         ->orderBy('ch.created', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				if ($date <= $row['created'])
					$data[$row['code']] = $row;
			}

			if ($data) {
				$this->loadCurrent($data, $date);

				return $data;
			} else {
				return $this->loadCurrent($data, $date);
			}
		} else {
			return $this->loadCurrent($data);
		}
	}

	protected function createProperty($row): Property
	{
		return new Property([
			'code'    => $row['code'],
			'home'    => $row['rate'],
			'rate'    => $row['rate'],
			'foreign' => 1,
		]);
	}

	protected function loadCurrent(array &$data, ?\DateTimeInterface $date = null)
	{
		$fromDb = $this->loadFromDB();
		$useCNB = [];

		foreach ($fromDb as $row) {
			if (!empty($data[$row['code']]))
				continue;

			if ($row['sync']) {
				$useCNB[$row['code']] = $row['code'];
			} else {
				$data[$row['code']] = $row;
			}
		}

		// Vytáhnutí s ČNB ty které se synchronizují a vytvoření historie pokud je potřeba
		if (!empty($useCNB)) {
			$this->loadFromCNB($data, $date);
		}

		// Pokud nenajde v ČNB tak použít tu z db
		foreach ($data as $code => $row) {
			if (empty($row))
				$data[$code] = $fromDb[$code];
		}

		return $data;
	}

	protected function loadFromDB(): array
	{
		if (is_null($this->cFromDB)) {
			foreach ($this->currenciesService->getAll() as $currency)
				$this->cFromDB[$currency->getCode()] = [
					'code' => $currency->getCode(),
					'home' => $currency->home,
					'rate' => $currency->rate,
					'id'   => $currency->getId(),
					'sync' => (bool) $currency->sync,
				];
		}

		return $this->cFromDB;
	}

	protected function loadFromCNB(array &$data, ?\DateTimeInterface $date = null)
	{
		$useCNB = [];
		foreach ($data as $code => $row)
			if (empty($row))
				$useCNB[] = $code;

		if (empty($useCNB))
			return;

		$tmp = parent::loadFromSource($date);
		if (!is_array($tmp))
			return;

		$fromDb = $this->loadFromDB();

		$sourceArray = [];
		foreach ($tmp as $row) {
			$arr                  = explode('|', $row);
			$sourceArray[$arr[3]] = [
				'code' => $arr[3],
				'home' => $arr[2],
				'rate' => $arr[4],
			];
		}

		foreach ($sourceArray as $row) {
			if (in_array($row['code'], $useCNB)) {
				$code  = $row['code'];
				$home  = $row['home'];
				$rate  = $row['rate'];
				$dbRow = $fromDb[$code];

				if (Config::load('default') !== $code)
					$rate = round($sourceArray[$code]['rate'] / $sourceArray[Config::load('default')]['rate'], 3);

				if ($dbRow['rate'] != $rate) {
					$history = new CurrencyHistory($this->em->getReference(Currency::class, $dbRow['id']),
						(float) $home, (float) $rate,
						$date ? DateTime::createFromFormat('Y-m-d H:i:s', $date->format('Y-m-d H:i:s')) : null);
					$this->em->persist($history)->flush();
				}

				$data[$code] = [
					'code' => $code,
					'home' => $home,
					'rate' => $rate,
				];
			}
		}
	}
}
