<?php declare(strict_types = 1);

namespace Currency\Model\Driver;

use Core\Model\Entities\EntityManagerDecorator;
use Currency\Model\Config;
use Currency\Model\Currencies;
use Currency\Model\Entities\CurrencyHistory;
use DateTime;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use h4kuna\Exchange\Currency\Property;
use h4kuna\Exchange\RatingList\RatingList;

class Database
{
	protected ?array $cFromDB = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Currencies             $currenciesService,
	)
	{
	}

	protected function getEmptyData(): array
	{
		$data = [];
		foreach ((array) Config::loadArray('whitelist') ?: [] as $v) {
			$data[$v] = [];
		}

		$data[Config::loadScalar('default')] = [
			'code' => Config::loadScalar('default'),
			'home' => 1,
			'rate' => 1,
		];

		return $data;
	}

	public function loadFromSource(?DateTime $date): RatingList
	{
		$data = $this->getEmptyData();

		if ($date instanceof DateTime) {
			foreach ($this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
				         ->select('c.code as code, ch.home, ch.rate, ch.created, c.id as id')
				         ->innerJoin('ch.currency', 'c')
				         ->where('ch.created >= :from')->andWhere('ch.created <= :to')
				         ->setParameters(new ArrayCollection([new Parameter('from', (clone $date)->setTime(0, 0, 0)), new Parameter('to', (clone $date)->setTime(23, 59, 59))]))
				         ->orderBy('ch.created', 'ASC')
				         ->getQuery()->getScalarResult() as $row) {
				/** @var array $row */
				if ($date <= $row['created']) {
					$data[$row['code']] = $row;
				}
			}

			if ($data !== []) {
				return $this->loadCurrent($data, $date);
			}

			return $this->loadCurrent($data, $date);
		}

		return $this->loadCurrent($data);
	}

	public function createProperty(array $row): Property
	{
		return new Property(
			(int) $row['home'],
			(float) $row['rate'],
			(string) $row['code'],
		);
	}

	protected function loadCurrent(array &$data, ?DateTimeInterface $date = null): RatingList
	{
		$fromDb = $this->loadFromDB();

		foreach ($fromDb as $row) {
			if (!empty($data[$row['code']])) {
				continue;
			}

			$data[$row['code']] = $row;
		}

		$properties = [];
		foreach ($data as $code => $row) {
			if (empty($row['home'])) {
				$row['home'] = 0;
			}
			if (empty($row['rate'])) {
				$row['rate'] = 1;
			}

			$properties[$code] = $this->createProperty($row);
		}

		return new RatingList(new DateTimeImmutable(), new DateTimeImmutable(), null, $properties);
	}

	protected function loadFromDB(): array
	{
		if (is_null($this->cFromDB)) {
			$this->cFromDB = [];

			foreach ($this->currenciesService->getAll() as $currency) {
				$this->cFromDB[$currency->getCode()] = [
					'code' => $currency->getCode(),
					'home' => $currency->home,
					'rate' => $currency->rate,
					'id'   => $currency->getId(),
					'sync' => (bool) $currency->sync,
				];
			}
		}

		return $this->cFromDB;
	}
}
