<?php declare(strict_types = 1);

namespace Currency\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\AdminModule\Model\Currencies;
use Currency\AdminModule\Model\Security;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use Exception;
use Nette\Application\ForbiddenRequestException;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use Nette\Utils\FileSystem;

class CurrencyForm extends BaseControl
{
	/** @var string|null @persistent */
	public ?string      $id       = null;
	protected ?Currency $currency = null;

	protected Currencies $currenciesService;
	protected Countries  $countriesService;
	protected Security   $security;
	protected Sites      $sitesService;

	public function __construct(
		Currencies $currencies,
		Countries  $countries,
		Security   $security,
		Sites      $sites
	)
	{
		$this->currenciesService = $currencies;
		$this->countriesService  = $countries;
		$this->security          = $security;
		$this->sitesService      = $sites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sites = [null => ''];
		foreach ($this->sitesService->getAll(false) as $site) {
			$sites[$site->getIdent()] = $site->getIdent();
		}

		$form->addBool('isActive', 'currency.currency.isActive')->setDefaultValue(1);
		$form->addText('code', 'currency.currency.code')->setRequired()
			->setMaxLength(8);
		$form->addText('symbol', 'currency.currency.symbol')->setRequired()
			->setMaxLength(10);
		$form->addInteger('decimals', 'currency.currency.decimals')->setRequired()
			->setHtmlAttribute('min', 0)
			->setHtmlAttribute('max', 3);
		$form->addBool('currencyFromLeft', 'currency.currency.currencyFromLeft')->setDefaultValue(0);
		$form->addText('title', 'currency.currency.title')->setRequired()
			->setMaxLength(255);
		$form->addText('nativeTitle', 'currency.currency.nativeTitle')
			->setMaxLength(255);
		$form->addSelect('site', 'currency.currency.eshop', $sites);
		$form->addSelect('country', 'currency.currency.country', ['' => ''] + $this->countriesService->getAllNameColumn());
		$form->addText('home', 'currency.currency.home')
			->setDefaultValue(1)
			->setNullable();
		$form->addText('rate', 'currency.currency.rate')
			->setRequired();
		$form->addBool('sync', 'currency.currency.sync');
		$form->addSelect('syncKey', 'currency.currency.syncKey', [
			''    => '',
			'ČNB' => 'ČNB',
		]);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		if ($this->id && !$this->security->edit() || !$this->id && !$this->security->new()) {
			throw new ForbiddenRequestException;
		}

		$this->em->beginTransaction();
		try {
			$currency = $this->id
				? $this->currenciesService->get($this->id)
				: new Currency($values->code, $values->title, $values->symbol, (float) $values->home, (float) $values->rate);

			if ($this->id && ($currency->rate != $values->rate || $currency->home != $values->home)) {
				$history = new CurrencyHistory($currency);
				$this->em->persist($history);
			}

			$currency->symbol           = $values->symbol;
			$currency->title            = $values->title;
			$currency->nativeTitle      = $values->nativeTitle;
			$currency->country          = $values->country ? $this->countriesService->getReference($values->country) : null;
			$currency->home             = Strings::formatEntityDecimal((float) $values->home);
			$currency->rate             = Strings::formatEntityDecimal((float) $values->rate);
			$currency->sync             = $values->sync;
			$currency->syncKey          = $values->syncKey;
			$currency->site             = $values->site ? $this->sitesService->getReference($values->site) : null;
			$currency->decimals         = $values->decimals;
			$currency->currencyFromLeft = (int) $values->currencyFromLeft;
			$currency->isActive         = $values->isActive;

			$this->em->persist($currency)->flush();
			$this->em->commit();
			FileSystem::delete(TMP_DIR . '/exchange');
			$this->presenter->flashMessageSuccess('default.saved');
			$form->addCustomData('currencyId', $currency->getId());

			$cache = new Cache($this->cacheStorage, \Currency\Model\Currencies::CACHE_NAMESPACE);
			$cache->clean([
				Cache::TAGS => ['currencies'],
			]);
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setCurrency(int $id): void
	{
		$this->currency = $this->currenciesService->get($id);

		if (!$this->currency) {
			throw new InvalidArgumentException;
		}

		$c = $this->currency;
		$f = $this['form'];
		$d = [
			'code'             => $c->getCode(),
			'symbol'           => $c->symbol,
			'title'            => $c->title,
			'nativeTitle'      => $c->nativeTitle,
			'home'             => $c->home,
			'rate'             => $c->rate,
			'sync'             => (int) $c->sync,
			'syncKey'          => $c->syncKey,
			'decimals'         => $c->decimals,
			'currencyFromLeft' => $c->currencyFromLeft,
			'is_active'        => $c->isActive,
		];

		if ($c->country && array_key_exists($c->country->getId(), $f->getComponent('country')->getItems())) {
			$d['country'] = $c->country->getId();
		}
		if ($c->site && array_key_exists($c->site->getIdent(), $f->getComponent('site')->getItems())) {
			$d['site'] = $c->site->getIdent();
		}

		$f->setDefaults($d);
		$f->getComponent('code')->setOmitted()->setDefaultValue($c->getCode())
			->getControlPrototype()->addClass('disabled');
	}
}
