<?php declare(strict_types = 1);

namespace Currency\FrontModule\Model\Subscribers;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Currency\Model\Config;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use Core\Model\Entities\EntityManagerDecorator;
use Tracy\Debugger;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Exchange $exchangeService;
	protected Currencies $currenciesService;
	protected EventDispatcher $eventDispatcher;

	public function __construct(EntityManagerDecorator $em, Exchange $exchange, Currencies $currencies, EventDispatcher $eventDispatcher)
	{
		$this->em                = $em;
		$this->exchangeService   = $exchange;
		$this->currenciesService = $currencies;
		$this->eventDispatcher   = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm' => ['createOrderForm', 100],
			'eshopOrders.orderBeforeSave' => ['orderBeforeSave', 100],
			'eshopOrders.orderOnSuccess'  => ['orderOnSuccess', 255],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$currency = $this->currenciesService->getCurrent();

		$event->form->getComponent('currency')->setDefaultValue($currency->getCode());
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values   = $event->formData;
		$currency = $this->currenciesService->getAll()[$values->currency] ?? null;

		if ($currency && $currency->getCode() !== $this->currenciesService->getDefaultCode()) {
			if ($currency->sync) {
				$rate = $this->exchangeService->change(1, $this->currenciesService->getDefaultCode(), $currency->getCode());
			} else {
				$rate = $currency->rate;
			}

			if (Config::load('logOrderCurrencyCreate')) {
				Debugger::log(Json::encode([
					'order'      => $event->order->getId(),
					'isSync'     => $currency->sync ? 'yes' : 'no',
					'rate'       => $rate,
					'entityRate' => $currency->rate,
				]), '_orderCurrency');
			}

			$entity           = new OrderCurrency($event->order, $values->currency);
			$entity->rate     = Strings::formatEntityDecimal($rate) ?? 0.0;
			$entity->decimals = $currency->decimals;

			$event->order->currency = $entity;
		}
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		$this->eventDispatcher->dispatch(new OrderEvent($event->order), Order::class . '::postLoad');
	}

}
