<?php declare(strict_types = 1);

namespace Currency\FrontModule\Model\Subscribers;

use EshopOrders\Model\EshopOrdersConfig;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;

class PaymentsSpeditionsSubscriber implements EventSubscriberInterface
{
	protected Exchange   $exchangeService;
	protected Currencies $currenciesService;

	public function __construct(Exchange $exchange, Currencies $currencies)
	{
		$this->exchangeService   = $exchange;
		$this->currenciesService = $currencies;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Payments::class . '::afterFillDao'   => ['paymentsAfterFillDao', -10],
			Speditions::class . '::afterFillDao' => ['speditionsAfterFillDao', -10],
		];
	}

	public function paymentsAfterFillDao(PaymentsSpeditionsEvent $event): void
	{
		$currency = $this->currenciesService->getCurrent();
		if (!$currency) {
			return;
		}

		foreach ($event->payments as $payment) {
			$payment->price = $this->exchangeService->change($payment->price, $currency->getCode());
			if ($payment->freeFrom !== null) {
				$payment->freeFrom = $this->exchangeService->change($payment->freeFrom, $currency->getCode());
			}
		}
	}

	public function speditionsAfterFillDao(PaymentsSpeditionsEvent $event): void
	{
		$currency = $this->currenciesService->getCurrent();
		if (!$currency) {
			return;
		}

		foreach ($event->speditions as $spedition) {
			$spedition->price = $this->exchangeService->change($spedition->price, $currency->getCode());

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				foreach ($spedition->weights as $weight) {
					$weight->price = $this->exchangeService->change($weight->price, $currency->getCode());
				}
			}

			if ($spedition->freeFrom !== null) {
				$spedition->freeFrom = $this->exchangeService->change($spedition->freeFrom, $currency->getCode());
			}
		}
	}

}
