<?php declare(strict_types = 1);

namespace Currency\FrontModule\Model\Subscribers;

use Core\Model\Event\Event;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\FrontModule\Model\Products;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ProductsSubscriber implements EventSubscriberInterface
{
	protected Exchange   $exchangeService;
	protected Currencies $currenciesService;

	public static array $used = [];

	public function __construct(
		Exchange   $exchange,
		Currencies $currencies
	)
	{
		$this->exchangeService   = $exchange;
		$this->currenciesService = $currencies;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Products::class . '::afterLoadFresh' => ['afterLoadFresh', -10],
			'eshopOrders.reloadCart'             => 'reloadCart',
		];
	}

	public function afterLoadFresh(ProductsEvent $event): void
	{
		$currency        = $this->currenciesService->getCurrent();
		$defaultCurrency = $this->currenciesService->getAll()[$this->currenciesService->getDefaultCode()] ?? null;
		$decimals        = $currency->decimals ?? ($defaultCurrency ? (int) $defaultCurrency->decimals : 0);

		if (!isset(self::$used[strtolower($currency->getCode())])) {
			self::$used[strtolower($currency->getCode())] = [];
		}

		foreach ($event->products as &$product) {
			$key = md5(implode('|', [(string) $product->getId(), (string) $product->price, (string) $product->priceInBaseCurrency]));

			if (in_array($key, self::$used[strtolower($currency->getCode())], true)) {
				continue;
			}


			// Vypocet do zakladni meny
			if ($product->currency && $product->currency !== $defaultCurrency->getCode()) {
				$product->basePriceInBaseCurrency = $this->exchangeService->change($product->basePriceInBaseCurrency, $defaultCurrency->getCode(), $product->currency);
				$product->priceInBaseCurrency     = $this->exchangeService->change($product->priceInBaseCurrency, $defaultCurrency->getCode(), $product->currency);

				if ($product->retailPriceInBaseCurrency) {
					$product->retailPriceInBaseCurrency = $this->exchangeService->change($product->retailPriceInBaseCurrency, $defaultCurrency->getCode(), $product->currency);
				}
			}

			$product->basePriceInBaseCurrency   = round($product->basePriceInBaseCurrency, $decimals);
			$product->priceInBaseCurrency       = round($product->priceInBaseCurrency, $decimals);
			$product->retailPriceInBaseCurrency = round($product->retailPriceInBaseCurrency, $decimals);

			// Vypocet pokud mena produktu je jina
			if (!$product->currency || $product->currency !== $currency->getCode()) {
				$product->basePrice = $this->exchangeService->change((float) $product->basePriceInBaseCurrency, $currency->getCode());
				$product->price     = $this->exchangeService->change((float) $product->priceInBaseCurrency, $currency->getCode());

				if ($product->retailPriceInBaseCurrency) {
					$product->retailPrice = $this->exchangeService->change((float) $product->retailPriceInBaseCurrency, $currency->getCode());
				}
			}

			$product->basePrice = round($product->basePrice, $decimals);
			$product->price     = round($product->price, $decimals);

			if ($product->retailPrice) {
				$product->retailPrice = round($product->retailPrice, $decimals);
			}

			self::$used[strtolower($currency->getCode())][$key] = $product->getId();
		}
	}

	public function reloadCart(Event $event): void
	{
		self::$used = [];
	}
}
