<?php declare(strict_types = 1);

namespace Currency\Model;

use Currency\Model\Driver\Cnb;
use DateTimeInterface;
use Exception;
use h4kuna\Exchange as h4kunaExchange;
use h4kuna\Exchange\Caching\Cache;
use h4kuna\Exchange\Driver\Driver;
use InvalidArgumentException;
use Nette\Utils\DateTime;

class Exchange
{
	public const DEFAULT = 'default';
	public const CURRENT = 'current';

	protected Config                  $config;
	protected Cache                   $cache;
	protected h4kunaExchange\Exchange $exchange;
	protected Driver                  $driver;
	protected Currencies              $currencies;

	public function __construct(Config $config, Driver $driver, Currencies $currencies)
	{
		$this->config     = $config;
		$this->driver     = $driver;
		$this->currencies = $currencies;
		$this->cache      = new Cache(TMP_DIR . '/exchange');
		$this->exchange   = new h4kunaExchange\Exchange($this->cache);

		$this->exchange->setDriver($driver);
		$this->exchange->setDefault((string) $config::load('default'));
	}

	public function changeByDate(float $price, DateTimeInterface $date, string $to = null, ?string $from = null): float
	{
		if (!in_array($to, $this->config->get('whitelist'))) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$toEntity = $this->currencies->getAll()[$to];
		$this->exchange->setDriver($toEntity->sync ? null : $this->driver, $date);

		return round($this->exchange->change($price, $from, $to), 3);
	}

	public function changeByDateReverse(float $price, DateTimeInterface $date, string $to = null, ?string $from = null): float
	{
		if (!in_array($to, $this->config->get('whitelist'))) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$fromEntity = $this->currencies->getAll()[$from];
		$this->exchange->setDriver($fromEntity->sync ? null : $this->driver, $date);

		return round($this->exchange->change($price, $from, $to), 3);
	}

	public function change(float $price, ?string $to = null, ?string $from = null): float
	{
		foreach (['to', 'from'] as $v) {
			if (${$v} === self::DEFAULT) {
				${$v} = $this->currencies->getDefaultCode();
			} else if (${$v} === self::CURRENT) {
				${$v} = $this->currencies->getCurrent()->getCode();
			}
		}

		if (!$to) {
			$to = $this->currencies->getCurrent()->getCode();
		}

		$toEntity = $this->currencies->getAll()[$to];

		if (!in_array($to, $this->config->get('whitelist'))) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$this->exchange->setDriver($toEntity->sync ? (new Cnb()) : $this->driver, new DateTime);

		try {
			$result = round($this->exchange->change($price, $from, $to), 5);
		} catch (Exception $e) {
			$this->cache->flushCache($this->driver);
			$result = round($this->exchange->change($price, $from, $to), 5);
		}

		return $result;
	}

	public function changeBySite(string $siteIdent, float $price, ?string $to = null, ?string $from = null): float
	{
		foreach (['to', 'from'] as $v) {
			if (${$v} === self::DEFAULT) {
				${$v} = $this->currencies->getDefaultCode();
			} else if (${$v} === self::CURRENT) {
				${$v} = $this->currencies->getCurrent()->getCode();
			}
		}

		if (!$to) {
			$to = $this->currencies->getCurrent()->getCode();
		}

		$toEntity = $this->currencies->getBySite($siteIdent)[$to] ?? $this->currencies->getCurrent()[$to];

		if (!in_array($to, $this->config->get('whitelist'))) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$this->exchange->setDriver($toEntity->sync ? (new Cnb()) : $this->driver, new DateTime);

		try {
			$result = round($this->exchange->change($price, $from, $to), 5);
		} catch (Exception $e) {
			$this->cache->flushCache($this->driver);
			$result = round($this->exchange->change($price, $from, $to), 5);
		}

		return $result;
	}
}
