<?php declare(strict_types = 1);

namespace Currency\FrontModule\Model\Subscribers;

use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Exchange;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CartSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected Exchange   $exchangeService,
		protected Currencies $currenciesService,
		protected Carts      $cartsService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.cartFillDaoItems' => ['onFillDaoItems', -10],
		];
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		if (!$this->cartsService->cDaoItems) {
			return;
		}

		$currency = $this->currenciesService->getCurrent();

		if ($currency instanceof Currency) {
			foreach ($this->cartsService->cDaoItems as $cartItem) {
				$this->processItem($cartItem, $currency);

				foreach ($cartItem->getChilds() as $child) {
					$this->processItem($child, $currency);
				}
			}
		}
	}

	protected function processItem(CartItem $cartItem, Currency $currency): void
	{
		$product = $cartItem->getProduct();
		$price   = (float) $cartItem->price;

		if ($cartItem->priceInBaseCurrency
			&& (!$product || !$product->currency || $product->currency !== $currency->getCode())
		) {
			$price = $this->exchangeService->change($cartItem->priceInBaseCurrency, $currency->getCode());
		}

		$price = round($price, $currency->decimals ?: 2);
		$cartItem->setPrice($price);
	}
}
