<?php declare(strict_types = 1);

namespace Currency\Model;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Http\Session;
use Core\Model\Sites;
use Core\Model\Templating\Filters\Price;
use Currency\FrontModule\Model\Subscribers\ProductsSubscriber;
use Currency\Model\Entities\Currency;
use Exception;
use Nette\Caching\Cache;
use Nette\Http\Request as HttpRequest;
use Nette\Http\SessionSection;
use Throwable;

class Currencies extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'currencies';

	/** @var string */
	protected $entityClass = Currency::class;

	/** @var Currency[][] */
	protected array $cAll = [];
	/** @var Currency[] */
	protected ?array         $cActive             = null;
	protected SessionSection $sessionSection;
	public static ?string    $currentCodeOverride = null;

	public function __construct(
		protected Sites           $sitesService,
		Session                   $session,
		protected HttpRequest     $httpRequest,
		protected EventDispatcher $eventDispatcher,
	)
	{
		$this->sessionSection = $session->getSection('currency');
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getDefaultCode(): string
	{
		$code = Config::load('default');

		return is_scalar($code) ? (string) $code : 'CZK';
	}

	/**
	 * @return Currency[]
	 * @throws Throwable
	 */
	public function getActive(): array
	{
		if ($this->cActive === null) {
			$this->cActive = [];
			foreach ($this->getAll() as $currency) {
				if ($currency->isActive) {
					$this->cActive[$currency->getCode()] = $currency;
				}
			}
		}

		return $this->cActive;
	}

	public function getBySite(string $siteIdent): array
	{
		try {
			if (!array_key_exists($siteIdent, $this->cAll)) {
				$this->cAll[$siteIdent] = [];

				foreach ((array) $this->getEr()->createQueryBuilder('c')
					->where('c.site = :currentSite OR c.site IS NULL')
					->setParameter('currentSite', $siteIdent)
					->getQuery()
					->getResult() ?: [] as $row) {
					/** @var Currency $row */
					$this->cAll[$siteIdent][$row->getCode()] = $row;
				}
			}
		} catch (Exception $e) {
			if (php_sapi_name() === 'cli') {
				return [];
			}

			throw $e;
		}

		return $this->cAll[$siteIdent];
	}

	/**
	 * @return Currency[]
	 * @throws Throwable
	 */
	public function getAll(): array
	{
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();

		try {
			if (!array_key_exists($siteIdent, $this->cAll)) {
				$this->cAll[$siteIdent] = [];

				foreach ((array) $this->getEr()->createQueryBuilder('c')
					->where('c.site = :currentSite OR c.site IS NULL')
					->setParameter('currentSite', $this->sitesService->getCurrentSite()->getIdent())
					->getQuery()
					->getResult() as $row) {
					/** @var Currency $row */
					$this->cAll[$siteIdent][$row->getCode()] = $row;
				}

				// TODO udelat cache pro currencues
				//				$this->cAll[$siteIdent] = $this->getCache()->load(self::CACHE_NAMESPACE . '_' . $siteIdent, function(&$dep) {
				//					$dep  = [
				//						Cache::EXPIRATION => '1 hour',
				//						Cache::Tags       => ['currencies'],
				//					];
				//					$data = [];
				//
				//					foreach ($this->getEr()->createQueryBuilder('c')
				//						         ->where('c.site = :currentSite OR c.site IS NULL')
				//						         ->setParameter('currentSite', $this->sitesService->getCurrentSite()->getIdent())
				//						         ->getQuery()
				//						         ->getResult() as $row) {
				//						$data[$row->getCode()] = $row;
				//					}
				//
				//					return $data;
				//				});
			}
		} catch (Exception $e) {
			if (php_sapi_name() === 'cli') {
				return [];
			}

			throw $e;
		}

		return $this->cAll[$siteIdent];
	}

	public function getCurrent(): ?Currency
	{
		// TODO vychozi podle eshopu
		$defaultFromConfig  = Config::load('frontEndDefault');
		$all                = $this->getAll();
		$defaultFromSession = $this->sessionSection->get('current');
		$current            = null;

		if (self::$currentCodeOverride) {
			self::$currentCodeOverride = strtoupper(self::$currentCodeOverride);
		}

		if (self::$currentCodeOverride && isset($all[self::$currentCodeOverride])) {
			return $all[self::$currentCodeOverride];
		}

		foreach ($all as $c) {
			if ($c->getCode() == $defaultFromSession || $c->getId() == $defaultFromSession) {
				return $c;
			}

			if ($c->getCode() == $defaultFromConfig || $c->getId() == $defaultFromConfig) {
				$current = $c;
			}
		}

		return $current ?: array_values($all)[0];
	}

	public function updatePriceFilter(Price $filter): void
	{
		if (is_string($_SERVER['REQUEST_URI']) && !str_starts_with($_SERVER['REQUEST_URI'], '/admin')) {
			$currency = $this->getCurrent();

			if (!$currency) {
				return;
			}

			$filter->setCurrencyFromLeft((bool) $currency->currencyFromLeft);
			$filter->setCurrency($currency->symbol);
			$filter->setDecimals($currency->decimals);
		}

		foreach ($this->getAll() as $currency) {
			$filter->addOtherCurrency($currency->getCode(), $currency->symbol, (bool) $currency->currencyFromLeft, $currency->decimals);
		}
	}

	public function setDefaultCurrency(): void
	{
		if (!$this->sessionSection->get('current')) {
			$domain     = $this->sitesService->getCurrentSite()->getCurrentDomain();
			$currencies = $this->getActive();

			if (!$domain) {
				return;
			}

			if ($domain->defaultCurrency && isset($currencies[$domain->defaultCurrency])) {
				$this->setCurrentCurrency($currencies[$domain->defaultCurrency]);
			}
		}
	}

	public function setCurrencyByUrl(): void
	{
		$urlCode = $this->httpRequest->getQuery('curr') ?: $this->httpRequest->getQuery('currency');

		if ($urlCode) {
			$currency = $this->getAll()[$urlCode] ?? null;

			if ($currency) {
				$this->setCurrentCurrency($currency);
			}
		}
	}

	public function setCurrentCurrency(Currency $currency): void
	{
		if (($currency->site && $currency->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent())) {
			return;
		}

		$event = new Event([
			'from' => $this->getCurrent()?->getCode() ?: $this->getDefaultCode(),
			'to'   => $currency->getCode(),
		]);
		$this->eventDispatcher->dispatch($event, 'eshopOrders.currencyChange');

		$this->sessionSection->set('current', $currency->getId());
	}

	public static function clearConvertedProducts(?string $currency = null): void
	{
		if ($currency) {
			ProductsSubscriber::$used[strtolower($currency)] = [];
		} else {
			ProductsSubscriber::$used = [];
		}
	}
}
