<?php declare(strict_types = 1);

namespace Currency\Model\Entities;

use DateTime;
use Core\Model\Entities\Country;
use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Currency\Model\Listeners\CurrencyListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table('currency__currency')]
#[ORM\Entity]
#[ORM\EntityListeners([CurrencyListener::class])]
class Currency
{
	use TId;

	#[ORM\Column(type: Types::STRING, length: 3, nullable: false)]
	private string $code;

	#[ORM\Column(type: Types::STRING, length: 10, nullable: false)]
	public string $symbol;

	#[ORM\JoinColumn(name: 'country', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\OneToOne(targetEntity: Country::class)]
	public ?Country $country = null;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $title;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $nativeTitle;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'home', type: Types::DECIMAL, precision: 10, scale: 5, nullable: false)]
	public $home;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'rate', type: Types::DECIMAL, precision: 10, scale: 5, nullable: false)]
	public $rate;

	/**
	 * @Gedmo\Timestampable(on="update")
	 */
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: false)]
	private ?DateTime $updated = null;

	/**
	 * @var Collection<CurrencyHistory>
	 */
	#[ORM\OneToMany(mappedBy: 'currency', targetEntity: CurrencyHistory::class, orphanRemoval: true)]
	#[ORM\OrderBy(['created' => 'DESC'])]
	private Collection $history;

	/**
	 * @var boolean
	 */
	#[ORM\Column(type: Types::BOOLEAN, nullable: true, options: ['default' => false])]
	public $sync;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $syncKey = null;

	#[ORM\JoinColumn(name: 'site_id', referencedColumnName: 'ident', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Site::class)]
	public ?Site $site = null;

	#[ORM\Column(type: Types::SMALLINT, length: 1, nullable: false, options: ['default' => 0])]
	public int $currencyFromLeft = 0;

	#[ORM\Column(type: Types::INTEGER, nullable: false, options: ['default' => 0])]
	public int $decimals = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $isActive = 1;

	public function __construct(string $code, string $title, string $symbol, float $home, float $rate)
	{
		$this->code             = $code;
		$this->title            = $title;
		$this->symbol           = $symbol;
		$this->home             = $home;
		$this->rate             = $rate;
		$this->sync             = false;
		$this->history          = new ArrayCollection;
		$this->currencyFromLeft = 0;
		$this->decimals         = 0;
		$this->isActive         = 1;
	}

	public function getCode(): string { return $this->code; }

	public function getUpdated(): ?DateTime { return $this->updated; }

	/** @return Collection<CurrencyHistory> */
	public function getHistory(): Collection { return $this->history; }
}
