<?php declare(strict_types = 1);

namespace Currency\Model;

use DateTime;
use Currency\Model\Driver\Database;
use DateTimeInterface;
use h4kuna\Exchange as h4kunaExchange;
use h4kuna\Exchange\Driver\Cnb\Day;
use h4kuna\Exchange\RatingList\CacheEntity;
use InvalidArgumentException;

class Exchange
{
	public const DEFAULT = 'default';
	public const CURRENT = 'current';

	protected h4kunaExchange\ExchangeFactory $exchangeFactory;

	public function __construct(
		protected Currencies $currencies,
		protected Database   $databaseDriver,
	)
	{
		/** @var string[] $whitelist */
		$whitelist = (array) Config::load('whitelist', []);

		$this->exchangeFactory = new h4kunaExchange\ExchangeFactory(
			from             : $this->currencies->getDefaultCode(),
			allowedCurrencies: $whitelist,
			tempDir          : TMP_DIR . '/exchange',
		);
	}

	public function getExchange(?string $from, ?string $to, bool $sync, ?DateTime $date = null): \h4kuna\Exchange\Exchange
	{
		if (!$from) {
			$from = $this->currencies->getDefaultCode();
		}

		if ($sync) {
			return $this->exchangeFactory->create(
				from       : $from,
				to         : $to,
				cacheEntity: new CacheEntity($date, new Day())
			);
		}

		return new h4kunaExchange\Exchange(
			from      : $from,
			ratingList: $this->databaseDriver->loadFromSource($date),
			to        : $to,
		);
	}

	public function changeByDate(float $price, DateTime $date, string $to = null, ?string $from = null): float
	{
		if (!in_array($to, (array) Config::load('whitelist'), true)) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$toEntity = $this->currencies->getAll()[$to];
		$exchange = $this->getExchange($from, $to, $toEntity->sync, $date);

		return round($exchange->change($price, $from, $to), 5);
	}

	public function changeByDateReverse(float $price, DateTimeInterface $date, string $to = null, ?string $from = null): float
	{
		if (!in_array($to, (array) Config::load('whitelist'), true)) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$fromEntity = $this->currencies->getAll()[$from];
		$exchange   = $this->getExchange($from, $to, $fromEntity->sync);

		return round($exchange->change($price, $from, $to), 3);
	}

	public function change(float $price, ?string $to = null, ?string $from = null): float
	{
		foreach (['to', 'from'] as $v) {
			if (${$v} === self::DEFAULT) {
				${$v} = $this->currencies->getDefaultCode();
			} else if (${$v} === self::CURRENT) {
				${$v} = $this->currencies->getCurrent()->getCode();
			}
		}

		if (!$to) {
			$to = $this->currencies->getCurrent()?->getCode() ?: $this->currencies->getDefaultCode();
		}

		$toEntity = $this->currencies->getAll()[$to];

		if (!in_array($to, (array) Config::load('whitelist'), true)) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$exchange = $this->getExchange($from, $to, $toEntity->sync);

		return round($exchange->change($price, $from, $to), 5);
	}

	public function changeBySite(string $siteIdent, float $price, ?string $to = null, ?string $from = null): float
	{
		foreach (['to', 'from'] as $v) {
			if (${$v} === self::DEFAULT) {
				${$v} = $this->currencies->getDefaultCode();
			} else if (${$v} === self::CURRENT) {
				${$v} = $this->currencies->getCurrent()->getCode();
			}
		}

		if (!$to) {
			$to = $this->currencies->getCurrent()?->getCode() ?: $this->currencies->getDefaultCode();
		}

		$toEntity = $this->currencies->getBySite($siteIdent)[$to] ?? $this->currencies->getCurrent();

		if (!in_array($to, (array) Config::load('whitelist'), true)) {
			throw new InvalidArgumentException("Currency '$to' is forbidden");
		}

		$exchange = $this->getExchange($from, $to, $toEntity->sync);

		return round($exchange->change($price, $from, $to), 5);
	}
}
