<?php declare(strict_types = 1);

namespace Currency\Model\Driver;

use Currency\Model\Config;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use DateTimeInterface;
use Exception;
use h4kuna\Exchange\Driver\Cnb\Day;
use h4kuna\Exchange\Driver\Cnb\Property;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Utils\DateTime;
use Tracy\Debugger;
use GuzzleHttp;

class Database extends Day
{
	protected EntityManagerDecorator $em;
	protected Currencies             $currenciesService;
	protected ?array                 $cFromDB = null;

	public function __construct(EntityManagerDecorator $em, Currencies $currencies)
	{
		$this->em                = $em;
		$this->currenciesService = $currencies;
	}

	protected function getEmptyData(): array
	{
		$data = [];
		foreach (Config::load('whitelist') as $v) {
			$data[$v] = [];
		}

		$data[Config::load('default')] = [
			'code' => Config::load('default'),
			'home' => 1,
			'rate' => 1,
		];

		return $data;
	}

	/**
	 * @throws Exception
	 */
	protected function loadFromSource(?DateTimeInterface $date): iterable
	{
		$this->setDate('Y-m-d', ($date ?: new DateTime())->format('Y-m-d'));
		$data = $this->getEmptyData();

		if ($date) {
			foreach ($this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
				         ->select('c.code as code, ch.home, ch.rate, ch.created, c.id as id')
				         ->innerJoin('ch.currency', 'c')
				         ->where('ch.created >= :from')->andWhere('ch.created <= :to')
				         ->setParameters([
					         'from' => (clone $date)->setTime(0, 0, 0),
					         'to'   => (clone $date)->setTime(23, 59, 59),
				         ])
				         ->orderBy('ch.created', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				if ($date <= $row['created']) {
					$data[$row['code']] = $row;
				}
			}

			if ($data) {
				$this->loadCurrent($data, $date);

				return $data;
			}

			return $this->loadCurrent($data, $date);
		}

		return $this->loadCurrent($data);
	}

	protected function createProperty($row): Property
	{
		/** @var array $row */
		return new Property([
			'code'    => $row['code'],
			'home'    => $row['rate'],
			'rate'    => $row['rate'],
			'foreign' => 1,
		]);
	}

	protected function loadCurrent(array &$data, ?DateTimeInterface $date = null): array
	{
		$fromDb = $this->loadFromDB();
		$useCNB = [];

		foreach ($fromDb as $row) {
			if (!empty($data[$row['code']])) {
				continue;
			}

			if ($row['sync']) {
				$useCNB[$row['code']] = $row['code'];
			} else {
				$data[$row['code']] = $row;
			}
		}

		// Vytáhnutí s ČNB ty které se synchronizují a vytvoření historie pokud je potřeba
		if (!empty($useCNB)) {
			$this->loadFromCNB($data, $date);
		}

		// Pokud nenajde v ČNB tak použít tu z db
		foreach ($data as $code => $row) {
			if (empty($row) && isset($fromDb[$code])) {
				$data[$code] = $fromDb[$code];
			}
		}

		return $data;
	}

	protected function loadFromDB(): array
	{
		if (is_null($this->cFromDB)) {
			$this->cFromDB = [];

			foreach ($this->currenciesService->getAll() as $currency) {
				$this->cFromDB[$currency->getCode()] = [
					'code' => $currency->getCode(),
					'home' => $currency->home,
					'rate' => $currency->rate,
					'id'   => $currency->getId(),
					'sync' => (bool) $currency->sync,
				];
			}
		}

		return $this->cFromDB;
	}

	protected function loadFromCNB(array &$data, ?DateTimeInterface $date = null): void
	{
		$logFile = '_CurrenciesLoadFromCNB';
		$useCNB  = [];
		foreach ($data as $code => $row) {
			if (empty($row)) {
				$useCNB[] = $code;
			}
		}

		Debugger::log('Start load from CNB for currencies:', $logFile);
		Debugger::log($useCNB, $logFile);

		if (empty($useCNB)) {
			return;
		}

		$tmp = parent::loadFromSource(DateTime::from($date));
		Debugger::log('loaded from source', $logFile);
		Debugger::log($tmp, $logFile);
		if (!is_array($tmp)) {
			return;
		}

		$fromDb = $this->loadFromDB();

		$sourceArray = [];
		foreach ($tmp as $row) {
			$arr                  = explode('|', $row);
			$sourceArray[$arr[3]] = [
				'code' => $arr[3],
				'home' => $arr[2],
				'rate' => $arr[4],
			];
		}

		foreach ($sourceArray as $row) {
			if (in_array($row['code'], $useCNB)) {
				$code  = $row['code'];
				$home  = $row['home'];
				$rate  = $row['rate'];
				$dbRow = $fromDb[$code];

				Debugger::log('Compare ', $logFile);
				Debugger::log($row, $logFile);

				if (Config::load('default') !== $code) {
					/** @var float $v1 */
					$v1 = $sourceArray[$code]['rate'];
					/** @var float $v2 */
					$v2   = $sourceArray[Config::load('default')]['rate'];
					$rate = round($v1 / $v2, 3);
				}

				Debugger::log('Rates ' . $dbRow['rate'] . ' != ' . $rate, $logFile);

				if ($dbRow['rate'] != $rate) {
					/** @var Currency $currency */
					$currency = $this->em->getReference(Currency::class, $dbRow['id']);

					$history = new CurrencyHistory(
						$currency,
						(float) $home, (float) $rate,
						$date ? DateTime::createFromFormat('Y-m-d H:i:s', $date->format('Y-m-d H:i:s')) : null
					);
					Debugger::log("Save rate code {$dbRow['id']}, home {$home}, rate {$rate}", $logFile);
					$this->em->persist($history)->flush();
				}

				$data[$code] = [
					'code' => $code,
					'home' => $home,
					'rate' => $rate,
				];
			}
		}
	}

	protected function createUrl(string $url, ?DateTimeInterface $date): string
	{
		if ($date === null) {
			return $url;
		}

		return $url . '?date=' . urlencode($date->format('d.m.Y'));
	}

	protected function downloadContent(?DateTimeInterface $date): string
	{
		$request = new GuzzleHttp\Client;

		return $request->request('GET', $this->createUrl('https://www.cnb.cz/cs/financni_trhy/devizovy_trh/kurzy_devizoveho_trhu/denni_kurz.txt', $date))->getBody()->getContents();
	}
}
