<?php declare(strict_types = 1);

namespace Currency\AdminModule\Components;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Currency\AdminModule\Model\Currencies;
use Currency\AdminModule\Model\Security;
use Currency\Model\Config;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use h4kuna\Exchange\Driver\Cnb;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;

class CurrenciesGrid extends BaseControl
{
	/** @var Currencies */
	protected $currenciesService;

	/** @var Security */
	protected $security;

	/** @var Config */
	protected $config;

	public function __construct(Currencies $currencies, Config $config, Security $security)
	{
		$this->currenciesService = $currencies;
		$this->config            = $config;
		$this->security          = $security;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleDelete($id)
	{
		if (!$this->security->delete())
			throw new ForbiddenRequestException();

		$presenter = $this->getPresenter();

		if ($this->currenciesService->remove($id))
			$presenter->flashMessageSuccess('default.deleted');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleSync(string $code): void
	{
		$presenter = $this->getPresenter();
		$currency  = $this->currenciesService->get($code);
		$list      = (new Cnb\Day())->download(null, $this->config->get('whitelist'));

		if (!$currency || !$list->offsetGet($code)) {
			$presenter->flashMessageDanger('currency.forbiddenCurrency');
		} else {
			$this->em->beginTransaction();
			try {
				$history = new CurrencyHistory($currency);
				$this->em->persist($history);
				$currency->rate = $list->offsetGet($code)->rate;
				$this->em->persist($currency);
				$this->em->flush();
				$this->em->commit();
				$this['grid']->reload();
				$presenter->flashMessageSuccess('currency.updated');
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();

				$presenter->flashMessageDanger('currency.updateError');
			}
		}

		$presenter->redrawControl('flashes');
	}

	public function handleHistory(string $code): void
	{
		$presenter = $this->getPresenter();
		$history   = $this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
			->where('ch.currency = :currency')->setParameter('currency', $code)
			->orderBy('ch.created', 'DESC')
			->getQuery()->getArrayResult();

		$this->template->history    = $history;
		$this->template->modalTitle = $this->translator->translate('currency.title.currencyHistory', null, ['currency' => $code]);
		$this->template->modal      = 'history';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->currenciesService->getEr()->createQueryBuilder('c')->addSelect('country')
			->leftJoin('c.country', 'country')
			->orderBy('c.title');
		$grid->setPrimaryKey('code');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('default', 'currency.currency.default')->setRenderer(function(Currency $row) {
			return $this->config->get('default') == $row->code
				? Html::el('span', ['class' => 'btn btn-xs btn-success'])->addHtml(Html::el('i', ['class' => 'fa fa-check']))
				: '';
		})->setAlign('center');
		$grid->addColumnText('code', 'currency.currency.code');
		$grid->addColumnText('symbol', 'currency.currency.symbol')->setAlign('center');
		$grid->addColumnText('title', 'currency.currency.title');
		$grid->addColumnText('nativeTitle', 'currency.currency.nativeTitle');
		$grid->addColumnText('country', 'currency.currency.country', 'country.name');
		$grid->addColumnNumber('rate', 'currency.currency.rate')->setFormat(3, ',', ' ');

		// Actions
		$grid->addAction('sync', '', 'sync!')->setIcon('fas fa-cloud-download-alt')->setBsType('success')
			->setRenderCondition(function(Currency $row) { return $row->sync; })->addClass('ajax');
		$grid->addAction('history', '', 'history!')->setIcon('fas fa-history')->setBsType('secondary')
			->addClass('ajax');
		if ($this->security->edit())
			$grid->addAction('edit', '', 'editCurrency!')->setIcon('fas fa-pencil-alt')->setBsType('primary')
				->addClass('ajax');
		if ($this->security->delete())
			$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
				->setConfirm('default.realltyDelete');

		// Prototype
		$grid->getColumn('default')->getElementPrototype('th')->class[] = 'w1nw';
		$grid->getColumn('symbol')->getElementPrototype('th')->class[]  = 'w1nw';
		$grid->getColumn('symbol')->getElementPrototype('td')->class[]  = 'w1nw';
		$grid->getColumn('code')->getElementPrototype('td')->class[]    = 'w1nw';

		return $grid;
	}
}
