<?php declare(strict_types = 1);

namespace Currency\AdminModule\Components;

use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\AdminModule\Model\Currencies;
use Currency\AdminModule\Model\Security;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;

class CurrencyForm extends BaseControl
{
	/** @var string @persistent */
	public $currencyCode;

	/** @var Currency */
	protected $currency;

	/** @var Currencies */
	protected $currenciesService;

	/** @var Countries */
	protected $countriesService;

	/** @var Security */
	protected $security;

	public function __construct(Currencies $currencies, Countries $countries, Security $security)
	{
		$this->currenciesService = $currencies;
		$this->countriesService  = $countries;
		$this->security          = $security;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('code', 'currency.currency.code')->setRequired()
			->setMaxLength(8);
		$form->addText('symbol', 'currency.currency.symbol')->setRequired()
			->setMaxLength(10);
		$form->addText('title', 'currency.currency.title')->setRequired()
			->setMaxLength(255);
		$form->addText('nativeTitle', 'currency.currency.nativeTitle')
			->setMaxLength(255);
		$form->addSelect('country', 'currency.currency.country', ['' => ''] + $this->countriesService->getAllNameColumn());
		$form->addText('home', 'currency.currency.home')->setType('number')->setAttribute('step', .01)
			->setDefaultValue(1)
			->setNullable();
		$form->addText('rate', 'currency.currency.rate')->setType('number')->setAttribute('step', .01)
			->setRequired();
		$form->addBool('sync', 'currency.currency.sync');
		$form->addSelect('syncKey', 'currency.currency.syncKey', [
			''    => '',
			'ČNB' => 'ČNB',
		]);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		if ($this->currencyCode && !$this->security->edit() || !$this->currencyCode && !$this->security->new())
			throw new ForbiddenRequestException();

		$this->em->beginTransaction();
		try {
			$currency = $this->currencyCode
				? $this->currenciesService->get($this->currencyCode)
				: new Currency($values->code, $values->title, $values->symbol, (float) $values->home, (float) $values->rate);

			if ($this->currency && $currency->code != $values->code && $this->currenciesService->get($values->code)) {
				$form->addError('currency.currency.codeExist');
				$this->redrawControl('form');

				return false;
			}

			if ($this->currencyCode && ($currency->rate != $values->rate || $currency->home != $values->home)) {
				$history = new CurrencyHistory($currency);
				$this->em->persist($history);
			}

			$currency->code        = strtoupper($values->code);
			$currency->symbol      = $values->symbol;
			$currency->title       = $values->title;
			$currency->nativeTitle = $values->nativeTitle;
			$currency->country     = $values->country ? $this->countriesService->getReference($values->country) : null;
			$currency->home        = (float) $values->home;
			$currency->rate        = (float) $values->rate;
			$currency->sync        = $values->sync;
			$currency->syncKey     = $values->syncKey;

			$this->em->persist($currency)->flush();
			$this->em->commit();
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$form->addCustomData('currencyCode', $currency->code);
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setCurrency(string $code)
	{
		$this->currency = $this->currenciesService->get($code);

		if (!$this->currency)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$c = $this->currency;
		$d = [
			'code'        => $c->code,
			'symbol'      => $c->symbol,
			'title'       => $c->title,
			'nativeTitle' => $c->nativeTitle,
			'home'        => $c->home,
			'rate'        => $c->rate,
			'sync'        => (int) $c->sync,
			'syncKey'     => $c->syncKey,
		];

		if ($c->country && array_key_exists($c->country->getId(), $this['form']['country']->getItems()))
			$d['country'] = $c->country->getId();

		$this['form']->setDefaults($d);
	}
}
