<?php declare(strict_types = 1);

namespace Currency\Model\Driver;

use Currency\Model\Config;
use Currency\Model\Entities\Currency;
use Currency\Model\Entities\CurrencyHistory;
use Doctrine\ORM\Query;
use h4kuna\Exchange\Driver\Cnb\Day;
use h4kuna\Exchange\Driver\Cnb\Property;
use h4kuna\Exchange\Driver\Driver;
use Kdyby\Doctrine\EntityManager;
use Nette\Utils\DateTime;

class Database extends Day
{

	/** @var EntityManager */
	protected $em;

	/** @var Config */
	protected $config;

	/** @var array */
	protected $cFromDB;

	public function __construct(EntityManager $em, Config $config)
	{
		$this->em     = $em;
		$this->config = $config;
	}

	protected function getEmptyData(): array
	{
		$data = [];
		foreach ($this->config->get('whitelist') as $v)
			$data[$v] = [];

		$data[$this->config->get('default')] = [
			'code' => $this->config->get('default'),
			'home' => 1,
			'rate' => 1,
		];

		return $data;
	}

	/**
	 * @param DateTime|NULL $date
	 *
	 * @return array
	 */
	protected function loadFromSource(?\DateTimeInterface $date): iterable
	{
		$this->setDate('Y-m-d', ($date ?: new DateTime())->format('Y-m-d'));
		$data = $this->getEmptyData();

		if ($date) {
			foreach ($this->em->getRepository(CurrencyHistory::class)->createQueryBuilder('ch')
				         ->select('IDENTITY(ch.currency) as code, ch.home, ch.rate, ch.created')
				         ->where('ch.created >= :from')->andWhere('ch.created <= :to')
				         ->setParameters([
					         'from' => (clone $date)->setTime(0, 0, 0),
					         'to'   => (clone $date)->setTime(23, 59, 59),
				         ])
				         ->orderBy('ch.created', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				if ($date <= $row['created'])
					$data[$row['code']] = $row;
			}

			if ($data) {
				$this->loadCurrent($data, $date);

				return $data;
			} else {
				return $this->loadCurrent($data, $date);
			}
		} else {
			return $this->loadCurrent($data);
		}
	}

	protected function createProperty($row): Property
	{
		return new Property([
			'code'    => $row['code'],
			'home'    => $row['rate'],
			'rate'    => $row['rate'],
			'foreign' => 1,
		]);
	}

	protected function loadCurrent(array &$data, ?\DateTimeInterface $date = null)
	{
		$fromDb = $this->loadFromDB();
		$useCNB = [];

		foreach ($fromDb as $row) {
			if (!empty($data[$row['code']]))
				continue;

			if ($row['sync']) {
				$useCNB[$row['code']] = $row['code'];
			} else {
				$data[$row['code']] = $row;
			}
		}

		// Vytáhnutí s ČNB ty které se synchronizují a vytvoření historie pokud je potřeba
		if (!empty($useCNB)) {
			$this->loadFromCNB($data, $date);
		}

		// Pokud nenajde v ČNB tak použít tu z db
		foreach ($data as $code => $row) {
			if (empty($row))
				$data[$code] = $fromDb[$code];
		}

		return $data;
	}

	protected function loadFromDB(): array
	{
		if (is_null($this->cFromDB))
			$this->cFromDB = $this->em->getRepository(Currency::class)->createQueryBuilder('c', 'c.code')
				->select('c.code, c.home, c.rate, c.sync')->getQuery()->getArrayResult();

		return $this->cFromDB;
	}

	protected function loadFromCNB(array &$data, ?\DateTimeInterface $date = null)
	{
		$useCNB = [];
		foreach ($data as $code => $row)
			if (empty($row))
				$useCNB[] = $code;

		if (empty($useCNB))
			return;

		$tmp = parent::loadFromSource($date);
		if (!is_array($tmp))
			return;

		$fromDb = $this->loadFromDB();
		foreach ($tmp as $row) {
			$row = explode('|', $row);
			if (in_array($row[3], $useCNB)) {
				$code  = $row[3];
				$home  = $row[2];
				$rate  = $row[4];
				$dbRow = $fromDb[$code];
				if ($dbRow['rate'] != $rate) {
					$history = new CurrencyHistory($this->em->getReference(Currency::class, $code),
						(float) $home, (float) $rate,
						$date ? DateTime::createFromFormat('Y-m-d H:i:s', $date->format('Y-m-d H:i:s')) : null);
					$this->em->persist($history)->flush();
				}

				$data[$code] = [
					'code' => $code,
					'home' => $home,
					'rate' => $rate,
				];
			}
		}
	}
}
