<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Events\GroupEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupText;
use DynamicModule\Model\Repository\Groups;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Tracy\Debugger;

class GroupForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int $groupId = null;

	/** @var string */
	protected string $moduleKey;

	/** @var Groups */
	protected Groups $groups;

	/** @var Group|null */
	protected ?Group $group = null;

	/**
	 * GroupForm constructor.
	 *
	 * @param string $moduleKey
	 * @param Groups $groups
	 */
	public function __construct(string $moduleKey, Groups $groups)
	{
		$this->moduleKey = $moduleKey;
		$this->groups    = $groups;
	}

	/**
	 * TODO pokud bylo nějaké tt typu image, tak pri ulozeni
	 * formu a nacteni jine polozky zustaly v tt image stejne data.
	 */
	protected function attached($presenter): void
	{
		if ($this->groupId) {
			$this->entity = $this->groups->get($this->groupId);
			$isGet        = $this->httpRequest->isMethod('GET');
			$q            = $this->httpRequest->getQuery('do');
			$postDo       = $this->httpRequest->getPost('_do');
			if ((!$isGet && $postDo && Strings::contains($postDo, 'uploadFromServerForm')) || ($isGet && $q && Strings::endsWith($q, 'showGallery')) || (!$isGet && $q && Strings::contains($q, 'galleryGallery')) ||
				(!$isGet && $q && (Strings::endsWith($q, 'Gallery-onEmpty') || Strings::endsWith($q, 'Gallery-upload')))) {
				$this->templateReader->loadTemplateComponents($this['form']['component'], $this->entity->template, $this->entity ? [$this->entity] : []);
			}
		}
		parent::attached($presenter);
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateGroup', TEMPLATES_DIR, ucfirst($this->moduleKey)));
		$this->templateReader->setTranslateKey(sprintf('%s.templateGroup', $this->moduleKey));

		$templates = $this->templateReader->getTemplates();
		$form      = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);
		if (DynamicModuleConfig::load('multiLangPublication')) {
			$form['isPublished']->setIsMultilanguage();
		}
		$form->addSelect('parent', 'default.parent', $this->getPossibleParents())
			->setTranslator(null);
		$form->addHidden('id', $this->group ? $this->group->getId() : null);
		$form->addSelect('template', $this->t('default.templates'), $templates)
			->setPrompt($this->t('default.choose'))
			->setRequired();
		$form->addComponent(new BaseContainer, 'component');

		$form->addSaveCancelControl()
			->closeModalOnCancel();

		$form->onValidate[] = function(BaseForm $form, ArrayHash $values) {
			$this->redrawControl('formErrors');
		};
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm  $form
	 * @param ArrayHash $values
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var GroupText[] $texts */
			$texts = [];

			if (!Validators::isNone($values->id)) {
				$this->group = $this->groups->get((int) $values->id);
			}

			if (!$this->group) {
				$this->group = new Group($this->moduleKey, $values->title);
			} else {
				$texts = $this->group->texts->toArray();
			}

			$this->group->title    = $values->title;
			$this->group->template = $values->template;

			if (DynamicModuleConfig::load('multiLangPublication')) {
				foreach ($langValues as $l => $v) {
					if (!isset($texts[$l]))
						$texts[$l] = new GroupText($this->group, $l);

					$texts[$l]->isPublished = (int) $v['isPublished'];

					$this->em->persist($texts[$l]);
				}
				$this->group->setEntityText($texts);
			} else {
				$this->group->isPublished = $values->isPublished;
			}

			if ($values->parent == '') {
				$this->group->setParent(null);
			} else {
				$this->group->setParent($this->groups->get($values->parent));
			}

			$this->group->setTexts($form->getHttpData()['component'] ?: []);

			$this->em->persist($this->group);
			$this->em->flush();

			$this->eventDispatcher->dispatch(new GroupEvent($this->group), 'dynamicModule.admin.' . $this->group->moduleKey . '.groupChanged');

			$form->addCustomData('groupId', $this->group->getId());
			$this->presenter->flashMessageSuccess('default.save');

			$cache = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);
		} catch (\Exception $e) {
			Debugger::log($e);
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	/**
	 * @param Group $group
	 */
	public function setGroup(Group $group): void
	{
		$this->groupId = $group->getId();
		$this->group   = $group;

		$form = $this['form'];
		$form->setSubmittedBy(null);
		$form->setDefaults([
			'title'       => $group->title,
			'isPublished' => $group->isPublished,
			'id'          => $group->getId(),
		]);
		$form['id']->setValue($group->getId());
		$form['parent']->setItems($this->getPossibleParents());

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$d = [];
			foreach ($this->group->getEntityTexts() as $l => $text) {
				$d['isPublished'][$l] = $text->isPublished;
			}

			if ($d) {
				$form->setDefaults($d);
			}
		}

		if ($this->group->parent && array_key_exists($this->group->parent->getId(), $form['parent']->getItems())) {
			$form['parent']->setDefaultValue($this->group->parent->getId());
		}

		if ($this->group->template && array_key_exists($this->group->template, $form['template']->getItems())) {
			$form['template']->setDefaultValue($this->group->template);

			if ($this->httpRequest->getQuery('do') != 'groupForm-loadInputs') {
				$this->templateReader->loadTemplateComponents($form['component'], $this->httpRequest->getPost('template') ?: $this->group->template, $this->entity ? [$this->entity] : []);
				$this->templateReader->setDefaults($form['component'], $this->group, $this->group->template);
			}
		}
	}

	/**
	 * @return array
	 */
	protected function getPossibleParents(): array
	{
		$groups = [null => ''];
		foreach ($this->groups->getQueryBuilderByModule($this->moduleKey)->orderBy('g.root')->addOrderBy('g.lvl')->addOrderBy('g.title')->getQuery()->getResult() as $g) {
			/** @var Group $g */
			if ($this->group && $this->group->getId() == $g->getId())
				continue;

			$arr  = [];
			$t    = $g;
			$skip = false;
			while ($t->getParent()) {
				$t = $t->getParent();
				if ($this->group && $this->group->getId() == $t->getId()) {
					$skip = true;
				}

				$arr[] = $t->title;
			}

			if ($skip)
				continue;

			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}

		return $groups;
	}

}
