<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Events\GroupEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Repository\Groups;
use Nette\Utils\Html;

class GroupsGrid extends BaseControl
{
	/** @var string */
	protected string $moduleKey;

	/** @var Groups */
	protected Groups $groups;

	/** @var IGroupFormFactory */
	protected IGroupFormFactory $groupFormFactory;

	/** @var IGroupDetailFactory */
	protected IGroupDetailFactory $groupDetailFactory;

	/**
	 * GroupsGrid constructor.
	 *
	 * @param string              $moduleKey
	 * @param Groups              $groups
	 * @param IGroupFormFactory   $groupFormFactory
	 * @param IGroupDetailFactory $groupDetailFactory
	 */
	public function __construct(string $moduleKey, Groups $groups, IGroupFormFactory $groupFormFactory, IGroupDetailFactory $groupDetailFactory)
	{
		$this->moduleKey          = $moduleKey;
		$this->groups             = $groups;
		$this->groupFormFactory   = $groupFormFactory;
		$this->groupDetailFactory = $groupDetailFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = 'dynamicModule.title.addGroup';
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'groupForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed|null $id
	 */
	public function handleEdit($id): void
	{
		$group = $this->groups->get((int) $id);

		if (!$group)
			return;

		$this['groupForm']->setGroup($group);
		$this->template->modalTitle       = $group->title;
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'groupForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		$group = $this->groups->get((int) $id);

		if ($this->groups->remove($id, $newStatus)) {
			$this->eventDispatcher->dispatch(new GroupEvent($group), 'dynamicModule.admin.' . $group->moduleKey . '.groupRemoved');
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/**
	 * @param $id
	 */
	public function handleShowDetail($id): void
	{
		$group = $this->groups->get((int) $id);

		if (!$group)
			return;

		$this['groupDetail']->setGroup($group);
		$this->template->modalTitle = $group->title;
		$this->template->modal      = 'groupDetail';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	/**
	 * @return BaseDataGrid
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->groups->getQueryBuilderByModule($this->moduleKey)
				->orderBy('g.root')->addOrderBy('g.lvl')->addOrderBy('g.title'));

		// Columns
		$grid->addColumnLink('title', 'default.title', 'edit!')->setRenderer(function($row) {
			return Html::el('a', ['class' => 'ajax',
			                      'href'  => $this->link('edit!', $row->getId())])->setText($row->title);
		});
		$grid->addColumnText('parent', 'default.parent', 'parent.title');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
				->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
				->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
				->onChange[] = [$this, 'gridPublishChange'];
		}

		// Filter
		$grid->addFilterText('title', '');

		// Actions
		$grid->addAction('detail', 'dynamicModule.group.sortMembers', 'showDetail!')->setIcon('sort')->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			// Columns prototype
			$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';
		}

		return $grid;
	}

	/**
	 * @return GroupForm
	 */
	public function createComponentGroupForm(): GroupForm
	{
		$form = $this->groupFormFactory->create($this->moduleKey);

		$form['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEdit((int) $form->getCustomData('groupId'));
			$this->redrawControl('flashes');
			$this['grid']->reload();
		};
		$form['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		return $form;
	}

	/**
	 * @return GroupDetail
	 */
	public function createComponentGroupDetail(): GroupDetail
	{
		return $this->groupDetailFactory->create();
	}

	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		$group = $this->groups->get($id);
		if ($this->groups->setPublish($id, (int) $newStatus)) {
			$this->eventDispatcher->dispatch(new GroupEvent($group), 'dynamicModule.admin.' . $group->moduleKey . '.groupPublishChanged');
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}
