<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\Seo;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Entities\MemberText;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Tracy\Debugger;

class MemberForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int $memberId = null;

	/** @var string */
	protected string $moduleKey;

	/** @var Members */
	protected Members $members;

	/** @var Groups */
	protected Groups $groups;

	/** @var Member|null */
	protected ?Member $member = null;

	protected SeoContainer $seoContainer;

	public function __construct(string $moduleKey, Members $members, Groups $groups, SeoContainer $seoContainer)
	{
		$this->moduleKey    = $moduleKey;
		$this->members      = $members;
		$this->groups       = $groups;
		$this->seoContainer = $seoContainer;
	}

	/**
	 * TODO pokud bylo nějaké tt typu image, tak pri ulozeni
	 * formu a nacteni jine polozky zustaly v tt image stejne data.
	 */
	protected function attached($presenter): void
	{
		if ($this->memberId) {
			$this->entity = $this->members->get($this->memberId);
			$isGet        = $this->httpRequest->isMethod('GET');
			$q            = $this->httpRequest->getQuery('do');
			$postDo       = $this->httpRequest->getPost('_do');
			if ((!$isGet && $postDo && Strings::contains($postDo, 'uploadFromServerForm')) || ($isGet && $q && Strings::endsWith($q, 'showGallery')) || (!$isGet && $q && Strings::contains($q, 'galleryGallery')) ||
				(!$isGet && $q && (Strings::endsWith($q, 'Gallery-onEmpty') || Strings::endsWith($q, 'Gallery-upload')))) {
				$this->templateReader->loadTemplateComponents($this['form']['component'], $this->entity->template, $this->entity ? [$this->entity] : []);
			}
		}
		parent::attached($presenter);
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateMember', TEMPLATES_DIR, ucfirst($this->moduleKey)));
		$this->templateReader->setTranslateKey(sprintf('%s.templateMember', $this->moduleKey));

		$templates = $this->templateReader->getTemplates();

		$groups        = [];
		$sortColumn    = DynamicModuleConfig::load('memberForm.groupSortColumn', 'title');
		$sortDirection = DynamicModuleConfig::load('memberForm.groupSortDirection', 'ASC');
		foreach ($this->groups->getQueryBuilderByModule($this->moduleKey)->addOrderBy('g.' . $sortColumn, $sortDirection)->getQuery()->getResult() as $g) {
			$arr = [];
			$t   = $g;
			while ($t->parent) {
				$t     = $t->parent;
				$arr[] = $t->title;
			}
			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}

		if (DynamicModuleConfig::load('memberForm.groupSortUseAsort', true))
			asort($groups);

		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);
		if (DynamicModuleConfig::load('multiLangPublication')) {
			$form['isPublished']->setIsMultilanguage();
		}
		$form->addCheckboxList('group', 'default.group', $groups)
			->setTranslator(null)
			->setRequired();
		$form->addSelect('template', $this->t('default.templates'), $templates)
			->setPrompt($this->t('default.choose'))
			->setRequired();
		$form->addComponent(new BaseContainer, 'component');
		$form->addHidden('id', $this->member ? $this->member->getId() : null);

		if (DynamicModuleConfig::load('member.allowSeo'))
			$form->addComponent($this->seoContainer->getContainer(true), 'seo');

		$form->addSaveCancelControl()
			->closeModalOnCancel();

		$form->onValidate[] = function() {
			$this->redrawControl('formErrors');
		};
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm  $form
	 * @param ArrayHash $values
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var MemberText[] $texts */
			$texts = [];

			if (!Validators::isNone($values->id)) {
				$this->member = $this->members->get((int) $values->id);
			}

			if (!$this->member) {
				$this->member = new Member($values->title, $values->template);
			} else {
				$texts = $this->member->texts->toArray();
			}

			$defaultGroupIds = array_map(static function(Group $group): int {
				return $group->getId();
			}, $this->member->getGroups());

			// add to group
			foreach (array_diff($values->group, $defaultGroupIds) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->addGroup($group);
				}
			}

			// remove from group
			foreach (array_diff($defaultGroupIds, $values->group) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->removeGroup($group);
				}
			}

			$this->member->title    = $values->title;
			$this->member->template = $values->template;

			if (DynamicModuleConfig::load('multiLangPublication')) {
				foreach ($langValues as $l => $v) {
					if (!isset($texts[$l]))
						$texts[$l] = new MemberText($this->member, $l);

					$texts[$l]->isPublished = (int) $v['isPublished'];

					$this->em->persist($texts[$l]);
				}
				$this->member->setEntityText($texts);
			} else {
				$this->member->isPublished = $values->isPublished;
			}

			$this->member->setTexts($form->getHttpData()['component'] ?: []);

			if (DynamicModuleConfig::load('member.allowSeo')) {
				foreach ($langValues as $lang => $vals) {
					if (!isset($texts[$lang]))
						$texts[$lang] = new MemberText($this->member, $lang);

					$seo              = $texts[$lang]->seo ?: new Seo();
					$seo->title       = $vals['seo']['title'];
					$seo->description = $vals['seo']['description'];
					$this->em->persist($seo);
					$texts[$lang]->seo = $seo;
					$this->em->persist($texts[$lang]);
				}

				$this->member->setEntityText($texts);
			}

			$this->em->persist($this->member);
			$this->em->flush();

			$this->eventDispatcher->dispatch(new MemberEvent($this->member), 'dynamicModule.admin.' . $this->member->groups->first()->group->moduleKey . '.memberChanged');

			$form->addCustomData('memberId', $this->member->getId());
			$this->presenter->flashMessageSuccess('default.save');

			$cache = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);
		} catch (\Exception $e) {
			Debugger::log($e);
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	/**
	 * @param Member $member
	 */
	public function setMember(Member $member): void
	{
		$this->memberId = $member->getId();
		$this->member   = $member;

		$form = $this['form'];
		$form->setSubmittedBy(null);
		$form->setDefaults([
			'title'       => $member->title,
			'isPublished' => $member->isPublished,
			'template'    => $member->template,
			'id'          => $member->getId(),
			'group'       => array_map(static function(Group $group): int {
				return $group->getId();
			}, $member->getGroups()),
		]);
		$form['id']->setValue($member->getId());

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$d = [];
			foreach ($this->member->getEntityTexts() as $l => $text) {
				$d['isPublished'][$l] = $text->isPublished;
			}

			if ($d) {
				$form->setDefaults($d);
			}
		}

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$d = [];
			foreach ($member->getEntityTexts() as $lang => $text) {
				if (!$text->seo)
					continue;

				$d['seo']['title'][$lang]       = $text->seo->title;
				$d['seo']['description'][$lang] = $text->seo->description;
			}
			$form->setDefaults($d);
		}

		if ($this->member->template && array_key_exists($this->member->template, $form['template']->getItems())) {
			$form['template']->setDefaultValue($this->member->template);

			if ($this->httpRequest->getQuery('do') != 'memberForm-loadInputs') {
				$this->templateReader->loadTemplateComponents($form['component'], $this->httpRequest->getPost('template') ?: $this->member->template, $this->member ? [$this->member] : []);
				$this->templateReader->setDefaults($form['component'], $this->member, $this->member->template);
			}
		}
	}

}
