<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Kdyby\Doctrine\Dql\Join;
use Nette\Utils\Html;

class MembersGrid extends BaseControl
{
	/** @var string */
	protected $moduleKey;

	/** @var Members */
	protected $members;

	/** @var IMemberFormFactory */
	protected $memberFormFactory;

	/** @var Groups */
	protected $groups;

	/**
	 * MembersGrid constructor.
	 *
	 * @param string             $moduleKey
	 * @param Members            $members
	 * @param IMemberFormFactory $memberFormFactory
	 * @param Groups             $groups
	 */
	public function __construct(string $moduleKey, Members $members, IMemberFormFactory $memberFormFactory, Groups $groups)
	{
		$this->moduleKey         = $moduleKey;
		$this->members           = $members;
		$this->memberFormFactory = $memberFormFactory;
		$this->groups            = $groups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = 'dynamicModule.title.addMember';
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'memberForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed|null $id
	 */
	public function handleEdit($id): void
	{
		$member = $this->members->get((int) $id);

		if (!$member) {
			return;
		}

		$this['memberForm']->setMember($member);

		$this->template->modalTitle       = $member->title;
		$this->template->modal            = 'memberForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->getPresenter();
		$member = $this->members->get((int) $id);

		if ($this->members->remove($id)) {
			$this->eventDispatcher->dispatch(new MemberEvent($member), 'dynamicModule.admin.' . $member->groups->first()->group->moduleKey . '.memberRemoved');
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	/**
	 * @return BaseDataGrid
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDataSource($this->members->getQueryBuilderByModule($this->moduleKey));
		$groups = $this->groups->getQueryBuilderByModule($this->moduleKey)->getQuery()->getResult();

		// Columns
		$grid->addColumnLink('title', 'default.title', 'edit!')->setRenderer(function($row) {
			return Html::el('a', ['class' => 'ajax',
			                      'href'  => $this->link('edit!', $row->getId())])->setText($row->title);
		});
		$grid->addColumnText('group', 'default.group')->setRenderer(function($row) {
			return implode(', ', array_map(function($v) { return $v->title; }, $row->getGroups()));
		});
		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
				 ->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
				 ->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
				->onChange[] = [$this, 'gridPublishChange'];
		}

		// Filter
		$grid->addFilterText('title', '');
		$groupsFilter = ['' => ''];
		foreach ($groups as $group) {
			$groupsFilter[$group->getId()] = $group->title;
		}
		$grid->getColumn('group')->setFilterSelect($groupsFilter, 'g.id');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		return $grid;
	}

	/**
	 * @return MemberForm
	 */
	public function createComponentMemberForm(): MemberForm
	{
		$form = $this->memberFormFactory->create($this->moduleKey);

		$form['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEdit((int) $form->getCustomData('memberId'));
			$this->redrawControl('flashes');
			$this['grid']->reload();
		};
		$form['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		return $form;
	}

	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		$member = $this->members->get($id);
		if ($this->members->setPublish($id, (int) $newStatus)) {
			$this->eventDispatcher->dispatch(new MemberEvent($member), 'dynamicModule.admin.' . $member->groups->first()->group->moduleKey . '.memberPublishChanged');
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}
