<?php declare(strict_types=1);

namespace DynamicModule\FrontModule\Model;

use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;

class GroupQuery
{
	/** @var string */
	protected string $moduleKey;

	/** @var \Closure[] */
	protected array $filter = [];

	/** @var \Closure[] */
	protected array $select = [];

	/** @var string */
	protected string $lang;

	/**
	 * GroupQuery constructor.
	 * @param string $moduleKey
	 */
	public function __construct(string $moduleKey)
	{
		$this->moduleKey = $moduleKey;
	}

	/**
	 * @param int[] $ids
	 */
	public function filterIds(array $ids): self
	{
		if ($ids) {
			$this->filter[] = function(QueryBuilder $qb) use ($ids) {
				if (count($ids) > 0) {
					$qb->andWhere('node.id IN (:ids)')->setParameter('ids', $ids);
				}
			};
		}

		return $this;
	}

	/**
	 * @param EntityRepository $repository
	 * @return QueryBuilder
	 */
	protected function doCreateQuery(EntityRepository $repository): QueryBuilder
	{
		$qb = $this->createBasicDql($repository);

		foreach ($this->select as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	/**
	 * @param EntityRepository $repository
	 * @return QueryBuilder
	 */
	protected function createBasicDql(EntityRepository $repository): QueryBuilder
	{
		$qb = $repository->createQueryBuilder('node')
						 ->select('node, IDENTITY(node.parent) as parent')
						 ->andWhere('node.moduleKey = :module')
						 ->orderBy('node.root, node.lft', 'ASC')
						 ->setParameter('module', $this->moduleKey);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('node.isPublished = 1');
		}

		foreach ($this->filter as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	/**
	 * @param EntityRepository $repository
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(EntityRepository $repository): QueryBuilder
	{
		return $this->doCreateQuery($repository);
	}

}