<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model\Repository;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\FrontModule\Model\Dao;
use DynamicModule\FrontModule\Model\GroupQuery;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use Nette\Caching\Cache;

class Groups extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'dynamicModuleGroups';

	/** @var string */
	protected $entityClass = Group::class;

	/** @var string */
	protected string $moduleKey;

	/** @var Fields */
	protected Fields $fields;

	/** @var Members */
	protected Members $members;

	/**
	 * Groups constructor.
	 *
	 * @param string  $moduleKey
	 * @param Fields  $fields
	 * @param Members $members
	 */
	public function __construct(string $moduleKey, Fields $fields, Members $members)
	{
		$this->moduleKey = $moduleKey;
		$this->fields    = $fields;
		$this->members   = $members;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param array $ids
	 *
	 * @return array
	 */
	public function getByIds(array $ids): array
	{
		$key = md5(implode('_', $ids));

		return $this->getCache()->load($key, function(&$dep) use ($ids) {
			$dep = [
				Cache::TAGS       => [self::CACHE_NAMESPACE],
				Cache::EXPIRATION => '1 week',
			];

			$repository = $this->getEr();

			$queryChild = new GroupQuery($this->moduleKey);
			if (!empty($ids))
				$queryChild->filterIds($ids);
			$childQb = $queryChild->getQueryBuilder($repository);

			$queryRoot = new GroupQuery($this->moduleKey);
			if (!empty($ids))
				$queryRoot->filterIds($ids);
			$rootQb = $queryRoot->getQueryBuilder($repository);

			if (DynamicModuleConfig::load('multiLangPublication')) {
				$childQb->join('node.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$childQb->setParameter('lang', $this->translator->getLocale());
				$rootQb->join('node.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$rootQb->setParameter('lang', $this->translator->getLocale());
			}

			if (!empty($ids))
				$rootQb->andWhere('node.id IN (:ids) AND (node.lvl = 0 OR (node.lvl = 1 AND node.parent NOT IN (:ids2)))')
					->setParameter('ids', $ids)
					->setParameter('ids2', $ids);

			return $this->buildTreeList($rootQb, $childQb);
		});
	}

	/**
	 * @param QueryBuilder $rootQb
	 * @param QueryBuilder $childQb
	 *
	 * @return array
	 */
	protected function buildTreeList(QueryBuilder $rootQb, QueryBuilder $childQb): array
	{
		$childQb->andWhere('node.lvl > 0')
			->orderBy('node.title'); // for children

		$tree     = [];
		$children = [];
		$groupIds = [];

		// prepare children
		$tmpChild = [];
		foreach ($childQb->getQuery()->getArrayResult() as $row) {
			$groupIds[] = $row[0]['id'];
			$tmpChild[] = $row;
		}

		// prepare root
		$tmpRoot = [];
		foreach ($rootQb->getQuery()->getArrayResult() as $row) {
			$groupIds[] = $row[0]['id'];
			$tmpRoot[]  = $row;
		}

		$fields  = $this->fields->getByGroups($groupIds);
		$members = $this->members->getByGroups($groupIds);

		// get all children
		foreach ($tmpChild as $row) {
			$parentId  = $row[0]['parent'] = $row['parent'];
			$moduleKey = $row[0]['moduleKey'];
			$id        = $row[0]['id'];

			$group                                = $this->fillDao($row[0]);
			$group->fields                        = $fields[$group->getId()] ?? [];
			$group->members                       = $members[$group->getId()] ?? [];
			$children[$moduleKey][$parentId][$id] = $group;
		}

		// get all roots and create tree
		foreach ($tmpRoot as $row) {
			$group     = $row[0];
			$moduleKey = $group['moduleKey'];

			$group['parent'] = null;
			$dao             = $this->fillDao($group);
			$dao->fields     = $fields[$dao->getId()] ?? [];
			$dao->members    = $members[$dao->getId()] ?? [];

			$children[$moduleKey][0][] = $dao;
			$tree[$moduleKey]          = $this->createTree($children[$moduleKey], $children[$moduleKey][0]);
		}

		$result = $tree[$moduleKey] ?? [];

		// set parent for second level children
		foreach ($result as $item) {
			foreach ($item->children as $ch) {
				$ch->parent = $item;
			}
		}

		return $result;
	}

	/**
	 * @param array $fields
	 *
	 * @return Dao\Group
	 */
	protected function fillDao(array $fields): Dao\Group
	{
		$group          = new Dao\Group((int) $fields['id'], $fields['title'], $fields['template']);
		$group->lvl     = (int) $fields['lvl'];
		$group->parent  = $fields['parent'];
		$group->fields  = [];
		$group->members = [];

		return $group;
	}

	/**
	 * @param Dao\Group[] $children
	 * @param Dao\Group[] $parents
	 *
	 * @return Dao\Group[]
	 */
	protected function createTree(array &$children, array $parents): array
	{
		$tree = [];
		foreach ($parents as $k => $l) {
			if (isset($children[$l->id])) {
				if ($l->lvl >= 1) {
					foreach ($children[$l->id] as $kk => $v) {
						$children[$l->id][$kk]->parent = &$parents[$k];
					}
				}
				$l->children = $this->createTree($children, $children[$l->id]);
			}

			$tree[$l->id] = $l;
		}

		return $tree;
	}

}
