<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model\Repository;

use Core\FrontModule\Model\Dao\Seo;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\QueryBuilder;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use DynamicModule\FrontModule\Model\Dao;
use Nette\Localization\ITranslator;

class Members extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Member::class;

	/** @var Fields */
	protected Fields $fields;

	protected array $cMembers = [];

	protected ITranslator $translator;

	/**
	 * Members constructor.
	 *
	 * @param Fields $fields
	 */
	public function __construct(Fields $fields, ITranslator $translator)
	{
		$this->fields     = $fields;
		$this->translator = $translator;
	}

	protected function joinSeo(QueryBuilder &$qb): void
	{
		if (DynamicModuleConfig::load('member.allowSeo')) {
			if (!in_array('txt', $qb->getAllAliases()))
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang')
					->setParameter('lang', $this->translator->getLocale());
			$qb->leftJoin('txt.seo', 'seo')
				->addSelect('seo.title as seoTitle, seo.description as seoDescription');
		}
	}

	protected function fillSeoDao(array $data): ?Seo
	{
		if (DynamicModuleConfig::load('member.allowSeo')) {
			$seo              = new Seo();
			$seo->title       = $data['seoTitle'] ?: null;
			$seo->description = $data['seoDescription'] ?: null;

			if ($seo->title && !Strings::contains($seo->title, '$siteName'))
				$seo->title = trim($seo->title . ' $separator $siteName');

			return $seo;
		}

		return null;
	}

	/**
	 * @param int $memberId
	 *
	 * @return Dao\Member|null
	 */
	public function getById(int $memberId): ?Dao\Member
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title')
			->andWhere('m.id = :memberId')
			->setParameter('memberId', $memberId);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		/** @var Dao\Member|null $row */
		$row = $qb->getQuery()->getOneOrNullResult();

		if (!$row) {
			return null;
		}

		$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
		$member->lang   = $row['lang'] ?? null;
		$member->fields = [];

		$seo = $this->fillSeoDao($row);
		if ($seo)
			$member->seo = $seo;

		$fields = $this->fields->getByMembers([$row['id']]);
		if (count($fields) > 0) {
			$member->fields = $fields[array_key_first($fields)];
		}

		return $member;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return Dao\Member[]
	 */
	public function getByIds(array $ids): array
	{
		$result = [];
		if (!$ids) {
			return [];
		}
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title')
			->andWhere('m.id IN(:memberIds)')
			->setParameter('memberIds', $ids);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->lang   = $row['lang'] ?? null;
			$member->fields = [];

			$seo = $this->fillSeoDao($row);
			if ($seo)
				$member->seo = $seo;

			$fields = $this->fields->getByMembers([$row['id']]);
			if (count($fields) > 0) {
				$member->fields = $fields[array_key_first($fields)];
			}
			$result[$member->getId()] = $member;
		}

		return $result;
	}

	/**
	 * @param int $groupId
	 *
	 * @return Member[]
	 */
	public function getByGroup(int $groupId): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.id = :groupId')
			->addOrderBy('gm.position')
			->setParameter('groupId', $groupId);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->lang   = $row['lang'] ?? null;
			$member->fields = [];

			$seo = $this->fillSeoDao($row);
			if ($seo)
				$member->seo = $seo;

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->fields = $v;
		}

		return $result;
	}

	/**
	 * @param string $module
	 * @param bool   $onlyPublished
	 *
	 * @return Dao\Member[]
	 */
	public function getByModule(string $module, bool $onlyPublished = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $module)
			->addOrderBy('m.title');

		if ($onlyPublished) {
			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere($qb->expr()->eq('m.isPublished', (int) $onlyPublished));
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->lang   = $row['lang'] ?? null;
			$member->fields = [];

			$seo = $this->fillSeoDao($row);
			if ($seo)
				$member->seo = $seo;

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->fields = $v;
		}

		return $result;
	}

	public function getByGroups(array $groupIds): array
	{
		$result = [];

		$structured     = [];
		$memberIds      = [];
		$membersForLoad = [];

		// vyhledani id clenu pro skupiny
		foreach ($this->em->getRepository(GroupMember::class)->createQueryBuilder('gm')
			         ->select('IDENTITY(gm.group) as group, IDENTITY(gm.member) as member')
			         ->orderBy('gm.position')
			         ->andWhere('gm.group IN (:groups)')
			         ->setParameter('groups', $groupIds)
			         ->getQuery()->getArrayResult() as $row) {
			$structured[$row['group']][] = $row['member'];
			$memberIds[]                 = $row['member'];

			if (!isset($this->cMembers[$row['member']]))
				$membersForLoad[] = $row['member'];
		}

		// nacteni zbyvajicich clenu
		if (!empty($membersForLoad)) {
			$qb = $this->getEr()->createQueryBuilder('m');
			$qb->select('m.id, m.lang, m.template, m.title, GROUP_CONCAT(IDENTITY(g.group)) as groups')
				->andWhere('m.id IN (:ids)')
				->leftJoin('m.groups', 'g')
				->groupBy('m.id')
				->setParameter('ids', array_unique($membersForLoad));

			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere('m.isPublished = 1');
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}

			$this->joinSeo($qb);

			$loaded = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
				$member->lang   = $row['lang'] ?? null;
				$member->fields = [];
				$member->setGroupIds(explode(',', (string) $row['groups']));

				$seo = $this->fillSeoDao($row);
				if ($seo)
					$member->seo = $seo;

				$loaded[$row['id']] = $member;
			}

			foreach ($this->fields->getByMembers(array_keys($loaded)) as $k => $v) {
				$loaded[$k]->fields = $v;
				$this->cMembers[$k] = $loaded[$k];
			}
		}

		// slozeni vysledku
		foreach ($structured as $groupId => $memberIds) {
			foreach ($memberIds as $memberId) {
				if (isset($this->cMembers[$memberId]))
					$result[$groupId][$memberId] = $this->cMembers[$memberId];
			}
		}

		return $result;
	}
}
