<?php declare(strict_types = 1);

namespace DynamicModule\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\TemplateReader\Entity\ITemplate;
use Doctrine\Common\Collections\ArrayCollection;
use DynamicModule\Model\Traits\TFields;
use DynamicModule\Model\Traits\TPublished;
use DynamicModule\Model\Traits\TTemplate;
use Gedmo\Mapping\Annotation as Gedmo;
use Doctrine\ORM\Mapping as ORM;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="dynamicmodule__group")
 * @ORM\Entity(repositoryClass="Gedmo\Tree\Entity\Repository\NestedTreeRepository")
 */
class Group implements ITemplate
{
	use TId;
	use TTemplate;

	/** @deprecated */
	use TPublished;

	use TFields;

	/**
	 * @var string
	 * @ORM\Column(type="string", length=255)
	 */
	public string $moduleKey;

	/**
	 * @var string
	 * @ORM\Column(type="string", length=255)
	 */
	public string $title;

	/**
	 * @var GroupMember[]
	 * @ORM\OneToMany(targetEntity="GroupMember", mappedBy="group", cascade={"all"})
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	public $members;

	/**
	 * @var ArrayCollection|Field[]
	 * @ORM\ManyToMany(targetEntity="DynamicModule\Model\Entities\Field", cascade={"persist"}, orphanRemoval=true)
	 * @ORM\JoinTable(name="dynamicmodule__group_fields")
	 */
	public $fields;

	/**
	 * @var Group[]
	 * @ORM\OneToMany(targetEntity="Group", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	public $children;

	/**
	 * @var Group|null
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Group", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE", nullable=true)
	 */
	public ?Group $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	public int $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	public int $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	public int $rgt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Group")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private Group $root;

	/**
	 * @var ArrayCollection|GroupText[]
	 * @ORM\OneToMany(targetEntity="GroupText", mappedBy="group", indexBy="lang")
	 */
	public $texts;

	/**
	 * Group constructor.
	 * @param string $moduleKey
	 * @param string $title
	 */
	public function __construct(string $moduleKey, string $title)
	{
		$this->title = $title;
		$this->moduleKey = $moduleKey;
		$this->members = new ArrayCollection;
		$this->children = new ArrayCollection;
		$this->fields = new ArrayCollection;
		$this->texts = new ArrayCollection;
	}

	/**
	 * @return GroupMember[]
	 */
	public function getMembers(): array
	{
		return $this->members->toArray();
	}

	/**
	 * @param Member $member
	 */
	public function addMember(Member $member): void
	{
		$this->members->add(new GroupMember($this, $member));
	}

	/**
	 * @return Group|null
	 */
	public function getRoot(): ?Group
	{
		return $this->root;
	}

	/**
	 * @param Group|null $parent
	 */
	public function setParent(?Group $parent = null): void
	{
		$this->parent = $parent;
	}

	/**
	 * @return Group|null
	 */
	public function getParent(): ?Group
	{
		return $this->parent;
	}

	/**
	 * @return Group[]
	 */
	public function getChildren(): array
	{
		return $this->children->toArray();
	}

	/**
	 * @return int
	 */
	public function getLevel(): int
	{
		return $this->lvl;
	}

	public function addEntityText(GroupText $text): self
	{
		$this->texts->add($text);

		return $this;
	}

	public function setEntityText(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	/**
	 * @return GroupText[]
	 */
	public function getEntityTexts(): array
	{
		$result = [];
		/** @var GroupText $text */
		foreach ($this->texts->toArray() as $text) {
			$result[$text->getLang()] = $text;
		}

		return $result;
	}

}