<?php declare(strict_types = 1);

namespace DynamicModule\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Gallery\FrontModule\Model\Images;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Pages\AdminModule\Model\TemplatePages;

class Group extends BaseNavigation implements INavigationItem
{
	protected $title           = 'group';
	protected $presenter       = 'DynamicModule:Front:Default';
	protected $action          = 'group';
	protected $presenterDetail = 'DynamicModule:Front:Default';
	protected $actionDetail    = 'member';

	protected ?string $navFormSite = null;

	protected TemplatePages $templatePagesService;

	protected Members $members;

	protected Groups $groups;

	protected Images $images;

	public function __construct(TemplatePages $templatePages, Members $members, Groups $groups, Images $images)
	{
		$this->templatePagesService = $templatePages;
		$this->members              = $members;
		$this->groups               = $groups;
		$this->images               = $images;
	}

	public function getGroup() { return 'Pages'; }

	public function navigationFormCustomHandle(NavigationForm $component, array $data = []): void
	{
		$this->navFormSite = $data['site'];
		$component->loadComponent($data['componentId']);
		$component->redrawControl('component');
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();
		$site      = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages     = $this->templatePagesService->getOptionsForSelectGrouped();

		$container->addSelect('module', 'dynamicModule.navigationForm.module', Helper::getAllModules());
		$container->addSelect('templatePage', 'dynamicModule.navigationForm.templatePage', $pages);
		$container->addBool('allowMemberDetail', 'dynamicModule.navigationForm.allowMemberDetail')
			->setDefaultValue(0);

		$baseForm->onAnchor[] = function() use ($container, $baseForm) {
			if (isset($baseForm->getValues()['component']['module'])) {
				$qb = $this->groups->getQueryBuilderByModule($baseForm->getValues()['component']['module'])
					->addOrderBy('g.parent')
					->addOrderBy('g.title');
				if (!DynamicModuleConfig::load('multiLangPublication')) {
					$qb->andWhere('g.isPublished = 1');
				}
				$groups = [];

				foreach ($qb->getQuery()->getResult() as $g) {
					$groups[] = [
						'id'     => $g->getId(),
						'parent' => $g->parent ? $g->parent->getId() : 0,
						'name'   => $g->title,
					];
				}

				$container->addCheckboxNestedList('baseGroup', 'dynamicModule.navigationForm.baseGroup', $groups)
					->setDefaultValue($baseForm->getParent()->navigation->componentParams['baseGroup']);
			}
		};

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return (
			$this->presenter == $presenter && $this->action == $action
			|| $this->presenterDetail == $presenter && $this->actionDetail == $action
		);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed|null
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$params = explode('/', $urlParams['path']);

		if (count($params) > 1) {
			$return = [
				'presenter' => $this->presenterDetail,
				'action'    => $this->actionDetail,
			];

			$checkedMember = $this->members->checkRoute(end($params), $navigation->componentParams['baseGroup'], $navigation->componentParams['module']);

			if (!$checkedMember) {
				$tmp = $params;
				array_shift($tmp);
				while (count($tmp) > 1 && !$checkedMember) {
					$checkedMember = $this->members->checkRoute(implode('/', $tmp), $navigation->componentParams['baseGroup'], $navigation->componentParams['module']);
					array_shift($tmp);
				}
			}

			if ($checkedMember)
				$return['id'] = (int) $checkedMember['id'];
			else {
				$id = explode('-', end($params), 2);

				if ($id[0] && $this->members->checkRoute($id[0], $navigation->componentParams['baseGroup'], $navigation->componentParams['module']))
					$return['id'] = (int) $id[0];
			}

			if (isset($return['id']))
				return $return;
		}

		if ($urlParams['path']) {
			$tmp = explode('/', $urlParams['path']);

			if (end($tmp) !== $navigation->alias)
				return null;
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['module'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if (isset($urlParams['id']) && is_numeric($urlParams['id'])) {
			if ($urlParams['action'] === $this->actionDetail) {
				$member = $this->members->checkRoute((int) $urlParams['id'], $navigation->componentParams['baseGroup'], $navigation->componentParams['module']);

				if ($member)
					return '/' . $navigation->alias . '/' . $member['url'];
				else
					return null;
			}

			if ($urlParams['action'] === $this->action) {
				$g = $this->groups->checkRoute((int) $urlParams['id'], $navigation->componentParams['module']);
				if ($g) {
					return '/' . $g['url'];
				}
			}
		}

		return $navigation->isHomepage ? '/' : '/' . $navigation->alias;
	}

	/**
	 * @param string             $lang
	 * @param ?DaoNavigationItem $navigation
	 * @param false              $siteMapData
	 *
	 * @return array
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		if (!$navigation || !$navigation->componentParams['allowMemberDetail'])
			return $arr;

		$members   = [];
		$galleries = [];
		foreach ($this->members->checkRoute(null, $navigation->componentParams['baseGroup'], $navigation->componentParams['module'], ['gallery']) as $id => $row) {
			if (!is_numeric($id))
				continue;

			$members[$id] = $row;

			if (isset($row['gallery']))
				$galleries[$row['gallery']] = $id;
		}

		if ($galleries) {
			foreach ($this->images->getImagesByAlbum(array_keys($galleries)) as $albumId => $album) {
				$memberId = $galleries[$albumId] ?? null;
				if (!$memberId)
					continue;

				$members[$memberId]['album'] = $album;
			}
		}

		foreach ($members as $id => $row) {
			if (!is_numeric($id))
				continue;

			$data = [
				'action'    => $this->actionDetail,
				'id'        => $id,
				'presenter' => $this->presenterDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col)
					$siteMapUrl->setSeo($col, $navigation->getSeo($col));

				if ($row['album']) {
					foreach ($row['album'] as $img)
						$siteMapUrl->addImage($img);
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$data['activeNavigation'] = $navigation;

			$arr[] = $data;
		}

		return $arr;
	}
}
