<?php declare(strict_types=1);

namespace DynamicModule\Model\Repository;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use Nette\Localization\Translator;
use Nette\Utils\Strings;

/**
 * Class Groups
 * @package DynamicModule\Model\Repository
 *
 * @method Group[] getAll()
 * @method Group|null get(int $id)
 * @method NestedTreeRepository getEr()
 */
class Groups extends BaseEntityService
{
	use TPublish;

	/** @var string */
	protected $entityClass = Group::class;

	protected array $checkRoute = [];

	/** @var Translator */
	protected Translator $translator;

	/**
	 * Groups constructor.
	 * @param Translator $translator
	 */
	public function __construct(Translator $translator)
	{
		$this->translator = $translator;
	}

	/**
	 * @param string $moduleKey
	 * @return QueryBuilder
	 */
	public function getQueryBuilderByModule(string $moduleKey): QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('g');
		$qb->andWhere('g.moduleKey = :module')
		   ->setParameter('module', $moduleKey);

		return $qb;
	}

	public function checkRoute(int $groupId, string $module): ?array
	{
		$qb = $this->getQueryBuilderByModule($module);
		$qb->select('g.id, g.title')
		   ->andWhere($qb->expr()->eq('g.id', $groupId));

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$qb->join('g.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$mKey = serialize($groupId . '_' . $module);
		if (!isset($this->checkRoute[$mKey])) {

			$group = $qb->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			if (!$group) {
				return null;
			}

			$this->checkRoute[$mKey] = [
				'id' => $group['id'],
				'title' => $group['title'],
				'url' => Strings::webalize($group['title']),
			];
		}
		return $this->checkRoute[$mKey];
	}
	
}