<?php declare(strict_types = 1);

namespace DynamicModule\Model\Repository;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Field;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use Nette\Localization\ITranslator;

/**
 * Class Members
 * @package DynamicModule\Model\Repository
 * @method Member[] getAll()
 * @method Member|null get(int $id)
 */
class Members extends BaseEntityService
{
	use TPublish;

	/** @var string */
	protected $entityClass = Member::class;

	protected ITranslator $translator;

	protected array $checkRoute = [];

	public function __construct(ITranslator $translator)
	{
		$this->translator = $translator;
	}

	/**
	 * @param string $moduleKey
	 *
	 * @return QueryBuilder
	 */
	public function getQueryBuilderByModule(string $moduleKey): QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->join('m.groups', 'gm', Join::WITH, 'm.id = gm.member')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $moduleKey);

		return $qb;
	}

	/**
	 * @param int $groupId
	 *
	 * @return QueryBuilder
	 */
	public function getQueryBuilderByGroup(int $groupId): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('gm')
			->from(GroupMember::class, 'gm')
			->join('gm.group', 'g')
			->where($qb->expr()->eq('g.id', $groupId))
			->addOrderBy('gm.position');

		return $qb;
	}

	public function checkRoute($key = null, array $groups, string $module, array $moreFields = [])
	{
		$mKey = $module . serialize($groups);
		if (!isset($this->checkRoute[$mKey])) {
			$checkRoute = [];

			$rows = $this->getEr()->createQueryBuilder('m')
						 ->select('m.id, m.title')
						 ->innerJoin('m.groups', 'gm', Join::WITH, 'gm.group IN (:groups)')
						 ->innerJoin('gm.group', 'g', Join::WITH, 'g.moduleKey = :module')
						 ->setParameter('groups', $groups)
						 ->setParameter('module', $module);

			if (DynamicModuleConfig::load('multiLangPublication')) {
				$rows->innerJoin('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$rows->setParameter('lang', $this->translator->getLocale());
			}

			foreach ($rows->getQuery()->getArrayResult() as $row)
				$checkRoute[$row['id']] = [
					'id'    => $row['id'],
					'title' => $row['title'],
				];

			if (!empty($checkRoute)) {
				$moreFields = array_merge($moreFields, ['title', 'alias', 'fullUrl']);

				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT mf.member_id, f.key, f.value FROM dynamicmodule__field f
    				INNER JOIN dynamicmodule__member_fields mf ON mf.field_id = f.id AND mf.member_id IN (" . implode(',', array_keys($checkRoute)) . ")
    				WHERE f.key IN ('" . implode('\',\'', $moreFields) . "') AND (f.lang = '{$this->translator->getLocale()}' OR f.lang IS NULL)") as $row) {
					$checkRoute[$row['member_id']][$row['key']] = $row['value'];
				}

				foreach ($checkRoute as $id => $vals) {
					if ($vals['fullUrl'] ?? null)
						$url = $vals['fullUrl'];
					else
						$url = $id . '-' . ($vals['alias'] ?? Strings::webalize($vals['title']));

					$checkRoute[$id]['url'] = $url;
					$checkRoute[$url]       = $checkRoute[$id];
				}
			}

			$this->checkRoute[$mKey] = $checkRoute;
		}

		return $key ? ($this->checkRoute[$mKey][$key] ?? null) : $this->checkRoute[$mKey];
	}
}
