<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\ArrayCollection;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\AdminModule\Model\FeatureValues;
use DynamicModule\Model\CacheService;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Feature;
use DynamicModule\Model\Entities\FeatureValue;
use DynamicModule\Model\Entities\FeatureValueText;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FeatureValueForm extends BaseControl
{
	public ?FeatureValue $featureValue = null;

	public function __construct(
		protected Features      $featuresService,
		protected FeatureValues $featureValues,
		protected CacheService  $cacheService
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function getFeature(): ?Feature
	{
		if ($this->featureValue) {
			return $this->featureValue->feature;
		}

		if ($this->getParameter('featureId')) {
			return $this->featuresService->get((int) $this->getParameter('featureId'));
		}

		return null;
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$features = $this->featuresService->getOptionsForSelect();

		$form->addSelect('feature', 'dynamicModule.featureValue.feature', $features)
			->setRequired();
		$form->addText('name', 'dynamicModule.featureValue.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addFilesManager('icon', 'dynamicModule.featureValue.icon')
			 ->setIsMultilanguage()
			 ->setMaxLength(255);
		$form->addBool('isPublished', 'dynamicModule.feature.isPublished')
			->setDefaultValue(1);

		if (DynamicModuleConfig::load('allowVirtualUrls')) {
			$form->addText('rodM', 'dynamicModule.featureValue.rodM')
				->setIsMultilanguage();
			$form->addText('rodZ', 'dynamicModule.featureValue.rodZ')
				->setIsMultilanguage();
			$form->addText('rodS', 'dynamicModule.featureValue.rodS')
				->setIsMultilanguage();
		}

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->featureValue) {
				$featureValue = $this->featureValue;
			} else {
				$featureValue = new FeatureValue;
			}

			$featureValue->feature = $this->featuresService->getReference($values->feature);

			$texts      = $featureValue->texts->toArray();
			$langValues = $form->convertMultilangValuesToArray();

			$this->em->persist($featureValue);

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] === '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text       = $texts[$lang] ?? new FeatureValueText($featureValue, $lang);
				$text->name = $vals['name'];
				$text->icon = $vals['icon'];

				if (DynamicModuleConfig::load('allowVirtualUrls')) {
					$text->rodM = $vals['rodM'];
					$text->rodZ = $vals['rodZ'];
					$text->rodS = $vals['rodS'];
				}

				$this->em->persist($text);

				$texts[$lang] = $text;
			}

			$featureValue->texts = new ArrayCollection($texts);

			$this->em->persist($featureValue);
			$this->em->flush();

			$form->addCustomData('featureValueId', $featureValue->getId());
			$form->addCustomData('featureId', $featureValue->feature->getId());

			$this->presenter->flashMessageSuccess('default.saved');

			$this->cacheService->removeFeature($featureValue->feature->getId());

			return true;
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setFeature(int $id): void
	{
		$this['form']->setDefaults([
			'feature' => $id,
		]);

		$this['form']->addCustomData('featureId', $id);
	}

	public function setFeatureValue(int $id): void
	{
		$this->featureValue = $this->em->getRepository(FeatureValue::class)->find($id);

		if (!$this->featureValue) {
			throw new InvalidArgumentException;
		}

		$form = $this['form'];
		$d    = [
			'feature'     => $this->featureValue->feature->getId(),
			'isPublished' => $this->featureValue->isPublished,
		];

		$form->setDefaults($d);

		$mTexts = $form->prepareMultilangTextsForForm([
			'name',
			'icon',
			'rodM',
			'rodZ',
			'rodS',
		], $this->featureValue->texts->toArray());
		$form->setDefaults($mTexts['texts']);
	}
}
