<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\AdminModule\Model\Events\GroupEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupText;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Nette\Application\AbortException;
use Nette\Caching\Cache;
use Nette\Utils\Html;

class GroupsGrid extends BaseControl
{
	public function __construct(
		protected string              $moduleKey,
		protected Groups              $groups,
		protected IGroupFormFactory   $groupFormFactory,
		protected IGroupDetailFactory $groupDetailFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = 'dynamicModule.title.addGroup';
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'groupForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed|null $id
	 */
	public function handleEdit($id): void
	{
		$group = $this->groups->get((int) $id);

		if (!$group) {
			return;
		}

		$this['groupForm']->setGroup($group);
		$this->template->modalTitle       = $group->title;
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'groupForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(mixed $id): void
	{
		$presenter = $this->presenter;
		$group     = $this->groups->get((int) $id);

		if ($this->groups->remove($id)) {
			$this->eventDispatcher->dispatch(
				new GroupEvent($group),
				'dynamicModule.admin.' . $group->moduleKey . '.groupRemoved',
			);
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleShowDetail(string|int $id): void
	{
		$group = $this->groups->get((int) $id);

		if (!$group) {
			return;
		}

		$this['groupDetail']->setGroup($group);
		$this->template->modalTitle = $group->title;
		$this->template->modal      = 'groupDetail';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource(
			$this->groups->getQueryBuilderByModule($this->moduleKey)
				->orderBy('g.lft'),
		);

		$grid->setSortable();
		$grid->setSortableHandler('groupsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'edit!')->setRenderer(fn(Group $row) => Html::el('a', [
			'class' => 'ajax',
			'href'  => $this->link('edit!', $row->getId()),
		])
			->setText(trim((string) (str_repeat('---', $row->getLevel() - 1) . ' ' . $row->title))));

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$grid->addColumnStatus('isPublished', 'default.isPublished')
				->setAlign('center')
				->addOption(1, 'default.publish')
				->setIcon('check')
				->setClass('btn-success')
				->setShowTitle(false)
				->endOption()
				->addOption(0, 'default.unPublish')
				->setIcon('times')
				->setClass('btn-danger')
				->setShowTitle(false)
				->endOption()
				->onChange[] = $this->gridPublishChange(...);
		} else {
			$grid->addColumnText('isPublished', 'default.isPublished')->setRenderer(function(Group $row) {
				$wrap = Html::el('div class=grid-langs');

				foreach ($this->langsService->getLangs(false) as $lang) {
					/** @var GroupText|null $text */
					$text = $row->texts->get($lang->getTag());
					$el   = Html::el('div class=grid-langs__lang');
					if ($text) {
						$a = Html::el('a', [
							'class' => 'ajax btn btn-xs ' . ($text->isPublished ? 'btn-success' : 'btn-danger'),
							'href'  => $this->link('invertPublish!', [$row->getId(), $lang->getTag()]),
						])->setText($lang->getShortTitle());
						$el->addHtml($a);
					}
					$wrap->addHtml($el);
				}

				return $wrap;
			});
		}

		// Filter
		$grid->addFilterText('title', '')->setCondition(function(QueryBuilder $qb, string $value) {
			$where = [];

			foreach (explode(' ', $value) as $k => $word) {
				if (!is_numeric($word) && strlen($word) < 2) {
					continue;
				}

				$where[] = 'g.title LIKE :title' . $k;
				$qb->setParameter('title' . $k, '%' . $word . '%');
			}

			if (!empty($where)) {
				$qb->andWhere('(' . implode(' AND ', $where) . ')');
			}
		});

		// Actions
		$grid->addAction('detail', 'dynamicModule.group.sortMembers', 'showDetail!')->setIcon('sort')->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			// Columns prototype
			$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';
		}

		$grid->setRowCallback(function(Group $row, $tr) {
			$groupId = ($row->getParent() ? $row->getParent()->getId() : '0') . '-' . $row->getLevel();
			/** @var Html $tr */
			$tr->addClass('group_' . $groupId);
			$tr->addAttributes(['data-group-tree' => $groupId]);

			return $tr;
		});

		return $grid;
	}

	public function createComponentGroupForm(): GroupForm
	{
		$form = $this->groupFormFactory->create($this->moduleKey);

		$form['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEdit((int) $form->getCustomData('groupId'));
			$this->redrawControl('flashes');
			$this['grid']->reload();
		};
		$form['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		return $form;
	}

	public function createComponentGroupDetail(): GroupDetail
	{
		return $this->groupDetailFactory->create();
	}

	/**
	 * @throws AbortException
	 */
	public function gridPublishChange(int|string $id, int|string $newStatus): void
	{
		$presenter = $this->presenter;

		$group = $this->groups->get((int) $id);
		if ($this->groups->setPublish($id, (int) $newStatus)) {
			$this->eventDispatcher->dispatch(
				new GroupEvent($group),
				'dynamicModule.admin.' . $group->moduleKey . '.groupPublishChanged',
			);
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$type      = $request->getPost('type');

		switch ($type) {
			case 'tree':
				$id    = $request->getPost('id');
				$move  = $request->getPost('move');
				$er    = $this->groups->getEr();
				$group = $this->groups->get($id);

				if ($id && is_numeric($move)) {
					if ($move < 0) {
						$er->moveUp($group, (int) abs((int) $move));
					} else if ($move > 0) {
						$er->moveDown($group, (int) $move);
					}

					$presenter->flashMessageSuccess('default.positionChanged');
				} else {
					$presenter->flashMessageDanger('default.positionChangeFailed');
				}

				$this->em->flush();

				$cache = new Cache(
					$this->cacheStorage,
					\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE
				);
				$cache->clean([Cache::Tags => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

				$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
				$cache->clean([Cache::Tags => Members::CACHE_NAMESPACE]);

				break;
		}

		$this->eventDispatcher->dispatch(new Event(), 'dynamicModule.admin.' . $this->moduleKey . '.membersSortChanged');

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->presenter;

		if ($this->groups->invertPublishText($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}
}
