<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\Model\Event\ControlEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\FrontModule\Model\Repository\Groups as FrontGroupsRepository;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Entities\MemberText;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Nette\Application\AbortException;
use Nette\Caching\Cache;
use Nette\Utils\Html;

class MembersGrid extends BaseControl
{
	protected FrontGroupsRepository $frontGroupsRepository;

	public function __construct(
		protected string             $moduleKey,
		protected Members            $members,
		protected IMemberFormFactory $memberFormFactory,
		protected Groups             $groups,
		protected IGroupsFactory     $groupsFactory,
	)
	{
		$this->frontGroupsRepository = $groupsFactory->create($moduleKey);
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = 'dynamicModule.title.addMember';
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'memberForm';
		$this->redrawControl('modal');
	}

	/**
	 * @param mixed|null $id
	 */
	public function handleEdit($id): void
	{
		$member = $this->members->get((int) $id);

		if (!$member) {
			return;
		}

		$this['memberForm']->setMember($member);

		$this->template->modalTitle       = $member->title;
		$this->template->modal            = 'memberForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleClone(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->members->cloneMember($id)) {
			$this->presenter->flashMessageSuccess('default.saved');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/**
	 * @throws AbortException
	 */
	public function handleDelete(
		mixed $id
	): void
	{
		$presenter = $this->presenter;
		$member    = $this->members->get((int) $id);

		if ($this->members->remove($id)) {
			$this->eventDispatcher->dispatch(
				new MemberEvent($member),
				'dynamicModule.admin.' . $member->groups->first()->group->moduleKey . '.memberRemoved',
			);
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDataSource($this->members->getQueryBuilderByModule($this->moduleKey));
		$groups = $this->groups->getOptionsForSelectNested($this->moduleKey);

		// Columns
		$grid->addColumnLink('title', 'default.title', 'edit!')->setRenderer(function($row) {
			return Html::el('a', [
				'class' => 'ajax',
				'href'  => $this->link('edit!', $row->getId()),
			])->setText($row->title);
		});
		$groupColumn = $grid->addColumnText('group', 'default.group')->setRenderer(function(Member $row) use ($groups) {
			$result = Html::el('div', [
				'style' => 'display: flex; flex-direction: column; gap: 4px;',
			]);

			foreach ($row->getGroups() as $v) {
				$result->addHtml(Html::el('div')->setText($groups[$v->getId()] ?? $v->title));
			}

			return $result;
		});
		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$grid->addColumnStatus('isPublished', 'default.isPublished')
				->setAlign('center')
				->addOption(1, 'default.publish')
				->setIcon('check')
				->setClass('btn-success')
				->setShowTitle(false)
				->endOption()
				->addOption(0, 'default.unPublish')
				->setIcon('times')
				->setClass('btn-danger')
				->setShowTitle(false)
				->endOption()
				->onChange[] = $this->gridPublishChange(...);
		} else {
			$grid->addColumnText('isPublished', 'default.isPublished')->setRenderer(function(Member $row) {
				$wrap = Html::el('div class=grid-langs');

				foreach ($this->langsService->getLangs(false) as $lang) {
					/** @var MemberText|null $text */
					$text = $row->texts->get($lang->getTag());
					$el   = Html::el('div class=grid-langs__lang');
					if ($text) {
						$a = Html::el('a', [
							'class' => 'ajax btn btn-xs ' . ($text->isPublished ? 'btn-success' : 'btn-danger'),
							'href'  => $this->link('invertPublish!', [$row->getId(), $lang->getTag()]),
						])->setText($lang->getShortTitle());
						$el->addHtml($a);
					}
					$wrap->addHtml($el);
				}

				return $wrap;
			});
		}

		// Filter
		$grid->addFilterText('title', '')->setCondition(function(QueryBuilder $qb, string $value) {
			$where = [];

			foreach (explode(' ', $value) as $k => $word) {
				if (!is_numeric($word) && strlen($word) < 2) {
					continue;
				}

				$where[] = 'm.title LIKE :title' . $k;
				$qb->setParameter('title' . $k, '%' . $word . '%');
			}

			if (!empty($where)) {
				$qb->andWhere('(' . implode(' AND ', $where) . ')');
			}
		});

		$groupColumn->setFilterSelect(['' => ''] + $groups, 'g.id');
		$groupColumn->getElementPrototype('td')->addAttributes(['style' => 'font-size: 13px; line-height: 1.3;']);

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('clone', '', 'clone!')->setIcon('copy')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setConfirm('default.reallyDelete')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax');

		$this->eventDispatcher->dispatch(new ControlEvent($grid), 'dynamicModule.admin.' . $this->moduleKey . '.membersGridAfterCreate');

		return $grid;
	}

	public function createComponentMemberForm(): MemberForm
	{
		$form = $this->memberFormFactory->create($this->moduleKey);

		$form['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEdit((int) $form->getCustomData('memberId'));
			$this->redrawControl('flashes');
			$this['grid']->reload();
		};
		$form['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		return $form;
	}

	/**
	 * @throws AbortException
	 */
	public function gridPublishChange(
		int|string $id, int|string $newStatus
	): void
	{
		$presenter = $this->presenter;

		$member = $this->members->get((int) $id);
		if ($this->members->setPublish($id, (int) $newStatus)) {
			$cache       = new Cache($this->cacheStorage, FrontGroupsRepository::CACHE_NAMESPACE);
			$memberCache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);

			$groups = [];
			foreach ($member->groups as $group) {
				$groups[] = $group->getId();
			}

			foreach ($this->langsService->getLangs(false) as $lang) {
				foreach ($groups as $groupId) {
					$cache->remove($this->moduleKey . '_' . md5((string) $groupId) . '_' . $lang->getTag());
				}

				$memberCache->remove('dynamicModuleMember/' . $lang->getTag() . '/' . $member->getId());
			}
			$cache->clean([Cache::TAGS => [FrontGroupsRepository::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);

			$this->eventDispatcher->dispatch(new MemberEvent($member), 'dynamicModule.admin.' . $member->groups->first()->group->moduleKey . '.memberPublishChanged');
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->presenter;

		if ($this->members->invertPublishText($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}
}
