<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\VirtualUrl;

use Core\AdminModule\Model\Sites;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\AdminModule\Model\FeatureValues;
use DynamicModule\AdminModule\Model\VirtualUrls;
use DynamicModule\FrontModule\Model\FilterUrlHelper;
use DynamicModule\Model\CacheService;
use DynamicModule\Model\Entities\VirtualUrl;
use DynamicModule\Model\Entities\VirtualUrlText;
use Exception;
use Navigations\AdminModule\Model\Navigations;
use Nette\Application\Attributes\Persistent;
use Nette\Caching\Cache;
use Nette\Forms\Controls\SubmitButton;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class VirtualUrlForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	public ?VirtualUrl $virtualUrl = null;

	public function __construct(
		public string             $siteIdent,
		protected VirtualUrls     $virtualUrls,
		protected Sites           $sites,
		protected Features        $features,
		protected FeatureValues   $featureValues,
		protected Navigations     $navigations,
		protected FilterUrlHelper $filterUrlHelper,
		protected CacheService    $cacheService
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$titlePrefix = 'dynamicModule.virtualUrl.';
		$form->addText('url', $titlePrefix . 'url')
			->setIsMultilanguage();
		$form->addText('title', $titlePrefix . 'title')
			->setIsMultilanguage();
		$form->addEditor('description', $titlePrefix . 'description')
			->setDisableAutoP(false)
			->setHeight(100)
			->setIsMultilanguage();

		$form->addText('seoTitle', $titlePrefix . 'seoTitle')
			->setIsMultilanguage();
		$form->addTextArea('seoDescription', $titlePrefix . 'seoDescription')
			->setIsMultilanguage();
		$form->addBool('addToSitemap', $titlePrefix . 'addToSitemap');

		$form->addSelect('navigation', $titlePrefix . 'navigation', $this->navigations->getOptionsForSelect())
			->setRequired();

		// Features
		$flat         = [];
		$featureTexts = $this->features->getOptionsForSelect();
		foreach ($this->featureValues->getOptionsForSelectGrouped() as $featureId => $values) {
			$flat[] = [
				'id'     => 'f' . $featureId,
				'name'   => $featureTexts[$featureId],
				'parent' => 0,
			];

			foreach ($values as $valueId => $value) {
				$flat[] = [
					'id'     => $valueId,
					'name'   => $value,
					'parent' => 'f' . $featureId,
				];
			}
		}
		$form->addCheckboxNestedList('features', $titlePrefix . 'features', $flat);

		$saveCancelControl = $form->addSaveCancelControl();

		/** @var SubmitButton $saveControl */
		$saveControl = $saveCancelControl->getComponent('save');
		$saveControl->setValidationScope([]);

		/** @var SubmitButton $saveAndCloseControl */
		$saveAndCloseControl = $saveCancelControl->getComponent('saveAndClose');
		$saveAndCloseControl->setValidationScope([]);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualUrlText[] $texts */
			$texts = [];

			$navigation = $this->navigations->get($values->navigation);
			if (!$navigation) {
				throw new Exception('Bad navigation');
			}

			$relationData = [
				FilterUrlHelper::$keyNav    => (int) $values->navigation,
				FilterUrlHelper::$keyValues => array_map(static fn($v) => (int) $v, $values->features),
			];

			$relationData = $this->filterUrlHelper->validateRelationData($relationData);
			if ($relationData === null) {
				throw new Exception('Bad relation data');
			}

			$relationHash = $this->filterUrlHelper->createRelationHash($this->siteIdent, $relationData);
			$oldUrls      = [];

			if ($this->id) {
				$virtualUrl = $this->virtualUrls->get($this->id);
				$texts      = $virtualUrl ? $virtualUrl->texts->toArray() : [];

				foreach ($texts as $v) {
					$oldUrls[$v->locale] = $v->url;
				}
			} else {
				$virtualUrl = new VirtualUrl($this->siteIdent, $relationHash, $navigation);
			}

			$virtualUrl->navigation   = $navigation;
			$virtualUrl->relationHash = $relationHash;

			foreach ($langValues as $l => $v) {
				if ($v['url'] && !str_starts_with($v['url'], '/')) {
					$v['url'] = '/' . $v['url'];
				}

				if (!isset($texts[$l]) && $v['url']) {
					$texts[$l] = new VirtualUrlText($virtualUrl, $l, $v['url']);
				}

				if (!$v['url']) {
					$this->em->remove($texts[$l]);

					unset($texts[$l]);
					continue;
				}

				foreach (['url', 'title', 'description', 'seoTitle', 'seoDescription'] as $col) {
					$texts[$l]->{$col} = $v[$col];
				}

				$this->em->persist($texts[$l]);
				$virtualUrl->texts->set($l, $texts[$l]);
			}

			// Features
			$formVals = [];
			foreach ($values->features as $value) {
				if (str_starts_with($value, 'f')) {
					continue;
				}

				$formVals[$value] = $value;
			}

			$entityVals = $virtualUrl->features->toArray();
			foreach (array_diff_key($formVals, $entityVals) as $key => $value) {
				$virtualUrl->features->set($key, $this->featureValues->getReference((int) $key));
			}

			foreach (array_diff_key($entityVals, $formVals) as $value) {
				$virtualUrl->features->removeElement($value);
			}

			$virtualUrl->addToSitemap = (int) $values->addToSitemap;

			$this->em->persist($virtualUrl);
			$this->em->flush();
			$this->em->commit();

			foreach ($langValues as $l => $v) {
				$cacheKey = $this->filterUrlHelper->getCacheKeyByHash(
					$this->siteIdent,
					$this->translator->getLocale(),
					$relationData
				);

				$this->cacheService->filterCache->remove($cacheKey);
			}

			foreach ($oldUrls as $l => $v) {
				$cacheKey = $this->filterUrlHelper->getCacheKeyByUrl(
					$this->siteIdent,
					$l,
					$v
				);

				$this->cacheService->filterCache->remove($cacheKey);
			}

			foreach ($virtualUrl->texts->toArray() as $v) {
				$cacheKey = $this->filterUrlHelper->getCacheKeyByUrl(
					$this->siteIdent,
					$v->locale,
					$v->url
				);

				$this->cacheService->filterCache->remove($cacheKey);
			}

			$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);

			$cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [Navigations::CACHE_NAMESPACE]]);

			$form->addCustomData('virtualUrlId', $virtualUrl->getId());
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function getVirtualUrl(): ?VirtualUrl
	{
		if (!$this->virtualUrl && $this->id) {
			$this->virtualUrl = $this->virtualUrls->get($this->id);
		}

		return $this->virtualUrl;
	}

	public function setVirtualUrl(int $id): void
	{
		$this->id = $id;
		$vu       = $this->virtualUrls->get($id);

		if (!$vu) {
			throw new InvalidArgumentException('');
		}

		$this->virtualUrl = $vu;

		$d = [
			'features'     => $vu->features->getKeys(),
			'navigation'   => $vu->navigation->getId(),
			'addToSitemap' => $vu->addToSitemap,
		];

		foreach ($vu->texts as $lang => $text) {
			$d['title'][$lang]          = $text->title;
			$d['description'][$lang]    = $text->description;
			$d['seoTitle'][$lang]       = $text->seoTitle;
			$d['seoDescription'][$lang] = $text->seoDescription;
			$d['url'][$lang]            = $text->url;
		}

		$this['form']->setDefaults($d);
	}
}
