<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\VirtualUrl;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\SaveCancelControl;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\AdminModule\Model\VirtualUrls;
use DynamicModule\Model\Entities\VirtualUrl;
use Nette\Utils\Html;

class VirtualUrlsGrid extends BaseControl
{
	public array $filter = [];

	public function __construct(
		protected string                     $siteIdent,
		protected VirtualUrls                $virtualUrls,
		protected IVirtualUrlFormFactory     $formFactory,
		protected IVirtualUrlTextFormFactory $textFormFactory
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============= Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('dynamicModule.title.addVirtualUrl');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id                 = $id;
		$this->template->modalTitle       = $this->t('dynamicModule.title.editVirtualUrl',
			['title' => $this['form']->getVirtualUrl()->getText()->title]);
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		if (is_array($id)
			? $this->virtualUrls->removeMultiple($id)
			: $this->virtualUrls->removeMultiple([$id])) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	public function handleTexts(int $id): void
	{
		$this['textForm']->id             = $id;
		$this->template->modalTitle       = $this->t('dynamicModule.title.editVirtualUrlTexts',
			['title' => $this['textForm']->getVirtualUrl()->getText()->title]);
		$this->template->modal            = 'textForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	/**
	 * @param int|string|array $id
	 */
	public function handleAddToSitemap($id): void
	{
		$presenter = $this->presenter;

		if (!is_array($id)) {
			$id = [$id];
		}

		$this->virtualUrls->addToSitemap($id);

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/**
	 * @param int|string|array $id
	 */
	public function handleRemoveFromSitemap($id): void
	{
		$presenter = $this->presenter;

		if (!is_array($id)) {
			$id = [$id];
		}

		$this->virtualUrls->removeFromSitemap($id);

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/*******************************************************************************************************************
	 * ============= Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$columnPrefix = 'dynamicModule.virtualUrl.';

		if (empty($this->filter)) {
			$grid->setDataSource([]);

			$grid->addColumnText('url', $columnPrefix . 'url', 'text.url');

			return $grid;
		}

		$qb = $this->virtualUrls->getEr()->createQueryBuilder('vu')
			->addSelect('t')
			->innerJoin('vu.texts', 't', Join::WITH, 't.locale = :locale')
			->andWhere('vu.siteIdent = :siteIdent')
			->setParameters(new ArrayCollection([new Parameter('locale', $this->translator->getLocale()), new Parameter('siteIdent', $this->siteIdent)]))
			->addOrderBy('vu.created', 'DESC');

		if (isset($this->filter['title']) && trim($this->filter['title']) !== '') {
			$qb->andWhere('t.title LIKE :title')
				->setParameter('title', '%' . $this->filter['title'] . '%');
		}

		if (isset($this->filter['hasTitle'])) {
			if ($this->filter['hasTitle'] === 1) {
				$qb->andWhere("t.title IS NOT NULL AND t.title != ''");
			} else if ($this->filter['hasTitle'] === 0) {
				$qb->andWhere("t.title IS NULL OR t.title = ''");
			}
		}

		if (isset($this->filter['inSitemap'])) {
			if ($this->filter['inSitemap'] === 1) {
				$qb->andWhere("vu.addToSitemap = 1");
			} else if ($this->filter['inSitemap'] === 0) {
				$qb->andWhere("vu.addToSitemap = 0");
			}
		}

		if ($this->filter['urlExact'] ?? null) {
			$qb->andWhere('t.url = :urlExact')
				->setParameter('urlExact', $this->filter['urlExact']);
		}

		if ($this->filter['urlContain'] ?? null) {
			$qb->andWhere('t.url LIKE :urlContain')
				->setParameter('urlContain', '%' . $this->filter['urlContain'] . '%');
		}

		if ($this->filter['navigations'] ?? null) {
			$qb->andWhere('vu.navigation IN (:navigations)')
				->setParameter('navigations', $this->filter['navigations']);
		}

		if ($this->filter['features'] ?? null) {
			$qb->innerJoin('vu.features', 'vuF', Join::WITH, 'vuF.id IN (:features)')
				->setParameter('features', $this->filter['features']);
		}

		$grid->setDataSource($qb);

		// Toolbar
		$grid->addToolbarButton('add!', 'default.add')
			->setIcon('plus')
			->setClass('ajax btn-success btn');

		// Columns
		$grid->addColumnText('url', $columnPrefix . 'url', 'text.url');
		$grid->addColumnText('title', $columnPrefix . 'title', 'text.title');
		$grid->addColumnText('inSitemap', $columnPrefix . 'inSitemap')
			->setRenderer(static function(VirtualUrl $row) {
				if ($row->addToSitemap) {
					return Html::el('span class="btn btn-success"')
						->addHtml(Html::el('i class="fa fa-check"'));
				}

				return '';
			})->setAlign('center');

		// Action
		$grid->addAction('edit', '', 'edit!', ['id' => 'id'])
			->setIcon('edit')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('texts', '', 'texts!', ['id' => 'id'])
			->setIcon('font')
			->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.delete')->onSelect[]                    = [$this, 'handleDelete'];
		$grid->addGroupAction($columnPrefix . 'addToSitemap')->onSelect[]      = [$this, 'handleAddToSitemap'];
		$grid->addGroupAction($columnPrefix . 'removeFromSitemap')->onSelect[] = [$this, 'handleRemoveFromSitemap'];

		// Prototype
		$grid->getColumn('inSitemap')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentForm(): VirtualUrlForm
	{
		$control = $this->formFactory->create($this->siteIdent);

		if ($this->getParameter('id')) {
			$control->setVirtualUrl((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function(BaseForm $baseForm) {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
				$this->handleEdit((int) $baseForm->getCustomData('virtualUrlId'));
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']->getComponent('saveControl');
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentTextForm(): VirtualUrlTextForm
	{
		$control = $this->textFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setVirtualUrl((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function(BaseForm $baseForm) {
				$this->presenter->redrawControl('flashes');
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']->getComponent('saveControl');
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}
}
