<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Components;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\FrontPresenter;
use DynamicModule\FrontModule\Model\Dao\FilterGroup;
use DynamicModule\FrontModule\Model\Dao\FilterItem;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\Dao\Member;
use DynamicModule\FrontModule\Model\FilterUrlHelper;
use DynamicModule\Model\CacheService;
use DynamicModule\Model\DynamicModuleConfig;
use Nette\Application\Attributes\Persistent;
use Nette\Caching\Cache;
use Nette\Http\Request;
use Nette\Http\Url;
use Nette\Utils\Strings;

class Filter extends BaseControl
{
	/** @var array<int, Member> */
	public array $members = [];

	/** @var array<int, Group> */
	public array $groups = [];

	public ?DaoNavigationItem $activeNav = null;

	public ?string $title = null;

	public array $onFilter = [];

	protected ?array $cFilters      = null;
	protected ?array $cFilterResult = null;

	#[Persistent]
	public array $filter     = [];
	public array $initFilter = [];

	public function __construct(
		protected Sites           $sites,
		protected Request         $httpRequest,
		protected FilterUrlHelper $filterUrlHelper,
		protected CacheService    $cacheService
	)
	{
		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter): void {
			if ($presenter->isAjax()) {
				foreach ($this->onFilter as $callback) {
					$callback();
				}

				if ($this->httpRequest->getUrl()->getQuery()) {
					$url = new Url($this->httpRequest->getUrl());
					$url->setQueryParameter('do', null);

					$this->presenter->payload->url = urldecode($url->getAbsoluteUrl());
				} else {
					$this->presenter->payload->url = urldecode($this->httpRequest->getUrl()->getAbsoluteUrl());
				}
			}
		});
	}

	public function render(): void
	{
		$this->initFilter = $this->filter;
		$this->parseInitFilters();

		$filters      = $this->applyFilters();
		$filterActive = false;

		foreach ($filters as $group) {
			if ($group->hasActive()) {
				$filterActive = true;
				break;
			}
		}

		$this->template->filters      = $filters;
		$this->template->filterActive = $filterActive;

		$this->template->render($this->getTemplateFile());
	}

	protected function parseInitFilters(): void
	{
		$this->initFilter = $this->initFilter ?? [];

		foreach ($this->initFilter as $k => $v) {
			if (is_array($v)) {
				$this->initFilter[$k] = array_map(static fn($a) => (int) $a, $v);
			} else {
				$this->initFilter[$k] = array_map(static fn($a) => (int) $a, explode('|', (string) $v));
			}
		}
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleSet(): void
	{
		if ($this->presenter->isAjax()) {
			foreach ($this->onFilter as $callback) {
				$callback();
			}

			if ($this->httpRequest->getUrl()->getQuery()) {
				$url = new Url($this->httpRequest->getUrl());
				$url->setQueryParameter('do', null);

				$this->presenter->payload->url = urldecode($url->getAbsoluteUrl());
			} else {
				$this->presenter->payload->url = urldecode($this->httpRequest->getUrl()->getAbsoluteUrl());
			}
		}
	}

	public function createLink(int $f, int $v): string
	{
		$this->filter = $this->initFilter;

		if ($f && $v) {
			if (isset($this->filter[$f])) {
				$i = array_search($v, $this->filter[$f]);

				if ($i !== false) {
					unset($this->filter[$f][$i]);

					if (empty($this->filter[$f])) {
						unset($this->filter[$f]);
					}
				} else {
					$this->filter[$f][] = $v;
				}
			} else {
				$this->filter[$f][] = $v;
			}
		}

		if (!empty($this->filter)) {
			foreach ($this->filter as $k => $v1) {
				sort($v1);
				$this->filter[$k] = $v1;
			}

			ksort($this->filter);
		}

		if (DynamicModuleConfig::load('allowVirtualUrls') && $this->activeNav && $this->filter) {
			$values = [];
			foreach ($this->filter as $v1) {
				foreach ($v1 as $v2) {
					$values[] = (int) $v2;
				}
			}

			$urlData = $this->filterUrlHelper->validateRelationData([
				FilterUrlHelper::$keyNav    => $this->activeNav->getId(),
				FilterUrlHelper::$keyValues => $values,
			]);

			if ($urlData) {
				$siteIdent = $this->sites->getCurrentSite()->getIdent();

				$cacheKey = $this->filterUrlHelper->getCacheKeyByHash(
					$siteIdent,
					$this->translator->getLocale(),
					$urlData
				);

				$cacheCheckExist = $this->cacheService->filterCache->load($cacheKey);

				if ($cacheCheckExist) {
					return $cacheCheckExist;
				} else {
					$relationHash = $this->filterUrlHelper->createRelationHash($siteIdent, $urlData);

					$exist = $this->em->getConnection()->fetchAssociative("SELECT vut.url 
						FROM dynamicmodule__virtual_url vu
						INNER JOIN dynamicmodule__virtual_url_text vut ON vu.id = vut.id AND vut.locale = :lang
						WHERE vu.relation_hash = :relationHash", [
						'lang'         => $this->translator->getLocale(),
						'relationHash' => $relationHash,
					]) ?: [];

					if (isset($exist['url'])) {
						$this->cacheService->filterCache->save($cacheKey, $exist['url'], [
							Cache::EXPIRATION => '30 minutes',
						]);

						return $exist['url'];
					}
				}

				$data = $this->filterUrlHelper->generate($this->title ?: '', $this->filter, array_values($this->groups)[0]);

				if (!$data['isMultiple']) {
					$baseUrl = $this->httpRequest->getUrl()->getPath();

					$tmp      = trim($baseUrl, '/');
					$tmp      = explode('/' . $this->translator->translate('default.urlPart.page'), $tmp)[0];
					$tmp      = explode('/', $tmp);
					$tmpAlias = array_pop($tmp);
					$lastPart = $this->title ?: $tmpAlias;
					$lastPart = trim($lastPart, '/');

					$lastPart = Strings::webalize($this->filterUrlHelper->addPrefixSuffix(
						$lastPart,
						$data['prefix'] . ' ',
						' ' . $data['suffix'])
					);
					$lastPart = trim($lastPart, '-');

					$tmp[] = $lastPart;

					$resultUrl = '/' . ltrim(implode('/', $tmp), '/');

					$relationHash = $this->filterUrlHelper->createRelationHash($siteIdent, $urlData);

					FilterUrlHelper::$checkExist[$relationHash] = [
						'url'          => $resultUrl,
						'relationHash' => $relationHash,
						'data'         => $urlData,
						'siteIdent'    => $siteIdent,
						'nav'          => $this->activeNav->getId(),
						'lang'         => $this->translator->getLocale(),
					];

					if ($resultUrl !== rtrim($this->httpRequest->getUrl()->getPath(), '/')) {
						$this->cacheService->filterCache->save($cacheKey, $resultUrl, [
							Cache::EXPIRATION => '30 minutes',
						]);
					}

					return $resultUrl;
				}
			}
		}

		foreach ($this->filter as $fk => $fv) {
			if (empty($fv)) {
				unset($this->filter[$fk]);
			} else {
				$this->filter[$fk] = implode('|', array_unique($fv));
			}
		}

		return urldecode($this->link('set!', ['filter' => $this->filter]));
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setMembers(array $members): void
	{
		$this->members = $members;
	}

	/**
	 * @return FilterGroup[]
	 */
	public function applyFilters(): array
	{
		if ($this->cFilters !== null) {
			return $this->cFilters;
		}

		if (empty($this->members)) {
			return [];
		}

		$this->cFilters = [];

		$activeValues = [];
		foreach ($this->filter ?? [] as $k => $v) {
			foreach (explode('|', (string) $v) as $av) {
				$activeValues[$k][(int) $av] = true;
			}
		}

		$filteredMembersCount = count($this->getMembersAfterFilter());
		foreach ($this->members as $member) {
			foreach ($member->features as $feature) {
				$featureId = $feature->feature->id;
				$valueId   = $feature->value->id;

				if (!isset($this->cFilters[$featureId])) {
					$tmp           = new FilterGroup($featureId, $feature->feature->name);
					$tmp->position = $feature->feature->position;

					$this->cFilters[$featureId] = $tmp;
				}

				$tmp                 = new FilterItem($valueId, $feature->value->name);
				$tmp->itemsCount     = $this->getFilterItemCount($featureId, $valueId);
				$tmp->icon           = $feature->value->icon;
				$tmp->addsItemsCount = empty($this->filter)
					? $tmp->itemsCount
					: max(0, $tmp->itemsCount - $filteredMembersCount);

				if (isset($activeValues[$featureId][$valueId])) {
					$tmp->isActive = true;
				}

				$this->cFilters[$featureId]->items[$valueId] = $tmp;
			}
		}

		uasort($this->cFilters, static function(FilterGroup $a, FilterGroup $b) {
			return $a->position <=> $b->position;
		});

		return $this->cFilters;
	}

	/** @return array<int, Member> */
	public function getMembersAfterFilter(): array
	{
		if ($this->cFilterResult === null) {
			$this->cFilterResult = $this->getFilteredItems($this->initFilter);;
		}

		return $this->cFilterResult;
	}

	public function getFilteredItems(array $filter): array
	{
		if (empty($filter)) {
			return $this->members;
		}

		$result = [];
		foreach ($this->members as $member) {
			$inFilters = [];
			foreach ($member->features as $feature) {
				$featureId = $feature->feature->id;
				$valueId   = $feature->value->id;

				if (isset($filter[$featureId]) && Arrays::contains($filter[$featureId], $valueId)) {
					$inFilters[$featureId][] = $valueId;
				}
			}

			if (
				count(array_diff_key($filter, $inFilters)) === 0 &&
				count(array_diff_key($inFilters, $filter)) === 0
			) {
				$result[$member->getId()] = $member;
			}
		}

		return $result;
	}

	public function getFilterItemCount(int $featureId, int $featureValueId): int
	{
		$appliedFilter             = $this->initFilter;
		$appliedFilter[$featureId] = [$featureValueId];

		$filteredItems = $this->getFilteredItems($appliedFilter);

		return count($filteredItems);
	}

	protected function resetFilter(): void
	{
		$this->filter     = [];
		$this->initFilter = [];
	}
}
