<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model\Repository;

use Contributte\Translation\Translator;
use Core\FrontModule\Model\Dao\Seo;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Doctrine\DBAL\Exception;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\FrontModule\Model\Dao;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use Nette\Caching\Cache;
use Nette\Utils\Json;

class Members extends BaseEntityService
{
	/** @var string */
	protected       $entityClass = Member::class;
	protected array $cMembers    = [];

	public function __construct(
		protected Fields     $fields,
		protected Translator $translator,
		protected Features   $features,
	)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, \DynamicModule\Model\Repository\Members::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	protected function joinSeo(QueryBuilder $qb): void
	{
		if (DynamicModuleConfig::load('member.allowSeo')) {
			if (!in_array('txt', $qb->getAllAliases(), true)) {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang')
					->setParameter('lang', $this->translator->getLocale());
			}

			$seoSelect = [];
			foreach (['title', 'description', 'robots', 'canonical', 'addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
				$seoSelect[] = "seo.{$col} as seo" . ucfirst($col);
			}

			$qb->leftJoin('txt.seo', 'seo')
				->addSelect(implode(',', $seoSelect));
		}
	}

	protected function fillSeoDao(array $data): Seo
	{
		$seo = new Seo();

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$seo->title       = $data['seoTitle'] ?: null;
			$seo->description = $data['seoDescription'] ?: null;

			foreach (['robots', 'canonical', 'addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
				if (array_key_exists('seo' . ucfirst($col), $data) && $data['seo' . ucfirst($col)] !== null) {
					$seo->{$col} = $data['seo' . ucfirst($col)];
				}
			}

			if ($seo->title && !Strings::contains($seo->title, '$siteName')) {
				$seo->title = trim($seo->title . ' $separator $siteName');
			}
		}

		return $seo;
	}

	/**
	 * @param Dao\Member[] $arr
	 *
	 * @return Dao\Member[]
	 */
	protected function postProcessDao(array $arr): array
	{
		foreach ($arr as $k => $v) {
			if (!$v->seo->title) {
				$template  = (string) explode('.', $v->getTemplate())[0];
				$seoFields = DynamicModuleConfig::load('member.defaultSeoFields.' . $v->module . '.' . $template) ?? [];

				if (array_key_exists('title', $seoFields)) {
					$v->seo->title = $v->getFieldValue($seoFields['title']) . ' $separator $siteName';
				}
			}
		}

		return $arr;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return Dao\Member[]
	 * @throws Exception
	 * @throws Exception
	 */
	public function getByIds(array $ids): array
	{
		$result = [];
		if (!$ids) {
			return [];
		}

		foreach ($ids as $id) {
			$keys[] = 'member/' . $this->translator->getLocale() . '/' . $id;
		}

		$whereIds = [];
		foreach ($this->getCache()->bulkLoad($keys) as $key => $member) {
			$tmp = explode('/', $key);

			if ($member) {
				$result[end($tmp)] = $member;
			} else {
				$whereIds[] = end($tmp);
			}
		}

		if (!empty($whereIds)) {
			$qb = $this->getEr()->createQueryBuilder('m');
			$qb->select('m.id, m.lang, m.template, m.title, m.params, g.moduleKey, GROUP_CONCAT(IDENTITY(gm.group)) as groups')
				->andWhere('m.id IN(:memberIds)')
				->join('m.groups', 'gm')
				->join('gm.group', 'g')
				->groupBy('m.id')
				->setParameter('memberIds', $ids);

			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere('m.isPublished = 1');
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}

			if (DynamicModuleConfig::load('allowFeatures')) {
				$qb->leftJoin('m.features', 'mf')
					->addSelect('GROUP_CONCAT(IDENTITY(mf.feature), \'|\', mf.id) as featureValuesIds');
			}

			$this->joinSeo($qb);

			$loaded = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
				$member->module = $row['moduleKey'];
				$member->lang   = $row['lang'] ?? null;
				$member->fields = [];
				$member->setGroupIds(explode(',', (string) $row['groups']));

				$member->seo = $this->fillSeoDao($row);

				if ($row['params']) {
					$member->params = is_array($row['params']) ? $row['params'] : Json::decode($row['params'], Json::FORCE_ARRAY);
				}

				if (DynamicModuleConfig::load('allowFeatures')) {
					$member->featureValueIds = $row['featureValuesIds'] ? $this->parseFeatureValues($row['featureValuesIds']) : [];
				}

				$fields = $this->fields->getByMembers([$row['id']]);
				if (count($fields) > 0) {
					$member->setFields($fields[array_key_first($fields)]);
				}
				$loaded[$member->getId()] = $member;
				$result[$member->getId()] = $member;
			}

			if (!empty($loaded)) {
				$this->postProcessDao($loaded);

				foreach ($loaded as $member) {
					$result[$member->getId()] = $member;

					$this->getCache()->save('member/' . $this->translator->getLocale() . '/' . $member->getId(), $member, [
						Cache::TAGS   => [\DynamicModule\Model\Repository\Members::CACHE_NAMESPACE],
						Cache::EXPIRE => '1 week',
					]);
				}
			}
		}

		$this->afterLoadDaoFromCache($result);

		return $result;
	}

	/**
	 * @return Dao\Member[]
	 * @throws Exception
	 */
	public function getByGroup(int $groupId): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title, g.moduleKey, m.params')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.id = :groupId')
			->groupBy('m.id')
			->addOrderBy('gm.position')
			->setParameter('groupId', $groupId);

		if (DynamicModuleConfig::load('allowFeatures')) {
			$qb->leftJoin('m.features', 'mf')
				->addSelect('GROUP_CONCAT(IDENTITY(mf.feature), \'|\', mf.id) as featureValuesIds');
		}

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('m.isPublished = 1');
		} else {
			$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
			$qb->setParameter('lang', $this->translator->getLocale());
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->lang   = $row['lang'] ?? null;
			$member->module = $row['moduleKey'];
			$member->fields = [];

			if ($row['params']) {
				$member->params = is_array($row['params']) ? $row['params'] : Json::decode($row['params'], Json::FORCE_ARRAY);
			}

			$member->seo = $this->fillSeoDao($row);

			if (DynamicModuleConfig::load('allowFeatures')) {
				$member->featureValueIds = $row['featureValuesIds'] ? $this->parseFeatureValues($row['featureValuesIds']) : [];
			}

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->setFields($v);
		}

		return $this->postProcessDao($result);
	}

	/**
	 * @return Dao\Member[]
	 * @throws Exception
	 */
	public function getByModule(string $module, bool $onlyPublished = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->select('m.id, m.lang, m.template, m.title, m.params')
			->join('m.groups', 'gm')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $module)
			->addOrderBy('m.title')
			->groupBy('m.id');

		if ($onlyPublished) {
			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere($qb->expr()->eq('m.isPublished', (int) $onlyPublished));
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $this->translator->getLocale());
			}
		}

		if (DynamicModuleConfig::load('allowFeatures')) {
			$qb->leftJoin('m.features', 'mf')
				->addSelect('GROUP_CONCAT(IDENTITY(mf.feature), \'|\', mf.id) as featureValuesIds');
		}

		$this->joinSeo($qb);

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
			$member->module = $module;
			$member->lang   = $row['lang'] ?? null;
			$member->fields = [];

			if ($row['params']) {
				$member->params = is_array($row['params']) ? $row['params'] : Json::decode($row['params'], Json::FORCE_ARRAY);
			}

			$member->seo = $this->fillSeoDao($row);

			if (DynamicModuleConfig::load('allowFeatures')) {
				$member->featureValueIds = $row['featureValuesIds'] ? $this->parseFeatureValues($row['featureValuesIds']) : [];
			}

			$result[$row['id']] = $member;
		}

		foreach ($this->fields->getByMembers(array_keys($result)) as $k => $v) {
			$result[$k]->setFields($v);
		}

		return $this->postProcessDao($result);
	}

	public function getByGroups(array $groupIds, ?string $lang = null): array
	{
		$result = [];
		$lang   = $lang ?: $this->translator->getLocale();

		$structured     = [];
		$memberIds      = [];
		$membersForLoad = [
			$lang => [],
		];

		if (!isset($this->cMembers[$lang])) {
			$this->cMembers[$lang] = [];
		}

		// vyhledani id clenu pro skupiny
		foreach ($this->em->getRepository(GroupMember::class)->createQueryBuilder('gm')
			         ->select('IDENTITY(gm.group) as group, IDENTITY(gm.member) as member')
			         ->orderBy('gm.position')
			         ->andWhere('gm.group IN (:groups)')
			         ->setParameter('groups', $groupIds)
			         ->getQuery()->getArrayResult() as $row) {
			$structured[$row['group']][] = $row['member'];
			$memberIds[]                 = $row['member'];

			if (!isset($this->cMembers[$lang][$row['member']])) {
				$membersForLoad[$lang][] = $row['member'];
			}
		}

		// nacteni zbyvajicich clenu
		if (!empty($membersForLoad[$lang])) {
			$qb = $this->getEr()->createQueryBuilder('m');
			$qb->select("m.id, m.lang, m.template, m.title, m.params, GROUP_CONCAT(IDENTITY(gm.group), '::', g.moduleKey) as groups")
				->andWhere('m.id IN (:ids)')
				->leftJoin('m.groups', 'gm')
				->join('gm.group', 'g')
				->groupBy('m.id')
				->setParameter('ids', array_unique($membersForLoad[$lang]));

			if (!DynamicModuleConfig::load('multiLangPublication')) {
				$qb->andWhere('m.isPublished = 1');
			} else {
				$qb->join('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
				$qb->setParameter('lang', $lang);
			}

			if (DynamicModuleConfig::load('allowFeatures')) {
				$qb->leftJoin('m.features', 'mf')
					->addSelect('GROUP_CONCAT(IDENTITY(mf.feature), \'|\', mf.id) as featureValuesIds');
			}

			$this->joinSeo($qb);

			$loaded = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$memberGroupsIds = [];
				$module          = null;

				foreach (explode(',', $row['groups']) as $v) {
					$tmp               = explode('::', $v);
					$memberGroupsIds[] = (int) $tmp[0];

					if (!$module) {
						$module = (string) $tmp[1];
					}
				}

				$member         = new Dao\Member($row['id'], $row['title'], $row['template']);
				$member->lang   = $row['lang'] ?? null;
				$member->module = $module;
				$member->fields = [];
				$member->setGroupIds($memberGroupsIds);

				if ($row['params']) {
					$member->params = is_array($row['params']) ? $row['params'] : Json::decode($row['params'], Json::FORCE_ARRAY);
				}

				if (DynamicModuleConfig::load('allowFeatures')) {
					$member->featureValueIds = $row['featureValuesIds'] ? $this->parseFeatureValues($row['featureValuesIds']) : [];
				}

				$member->seo = $this->fillSeoDao($row);

				$loaded[$row['id']] = $member;
			}

			if (!empty($loaded)) {
				foreach ($this->fields->getByMembers(array_keys($loaded)) as $k => $v) {
					$loaded[$k]->setFields($v);
				}

				$this->postProcessDao($loaded);

				$this->cMembers[$lang] = $loaded + $this->cMembers[$lang];
			}
		}

		// slozeni vysledku
		foreach ($structured as $groupId => $memberIds) {
			foreach ($memberIds as $memberId) {
				if (isset($this->cMembers[$lang][$memberId])) {
					$result[$groupId][$memberId] = $this->cMembers[$lang][$memberId];
				}
			}
		}

		return $result;
	}

	protected function parseFeatureValues(string $val): array
	{
		$result = [];

		foreach (explode(',', $val) as $row) {
			$tmp                   = explode('|', $row);
			$result[(int) $tmp[1]] = (int) $tmp[0];
		}

		return $result;
	}

	public function afterLoadDaoFromCache(array $members): void
	{
		if (DynamicModuleConfig::load('allowFeatures')) {
			$this->features->getForMembers($members);
		}
	}
}
