<?php declare(strict_types = 1);

namespace DynamicModule\Model\Console;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use DynamicModule\Model\DynamicModuleConfig;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'dynamic-module:fill-entity-text')]
class FillEntityText extends Command
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('dynamic-module:fill-entity-text')->setDescription('Add translation text which missed');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('WORKING...');

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$output->writeln('You must enable multiLangPublication in configuration');

			return Command::FAILURE;
		}

		$connection = $this->em->getConnection();
		try {
			$connection->transactional(function() use ($connection, $output) {
				$langs         = $this->translator->getLocalesWhitelist();
				$defaultLocale = $this->translator->getDefaultLocale();
				$groups        = $connection->fetchAllAssociative("SELECT id FROM dynamicmodule__group");
				$groupTexts    = $connection->fetchAllAssociative("SELECT group_id id, lang FROM dynamicmodule__group_text");
				$members       = $connection->fetchAllAssociative("SELECT id FROM dynamicmodule__member");
				$memberTexts   = $connection->fetchAllAssociative(
					"SELECT member_id id, lang, seo_id FROM dynamicmodule__member_text",
				);
				$isMissingText = false;

				foreach ($groups as $g) {
					$groupId = $g['id'];
					$texts   = [];
					foreach ($groupTexts as $gt) {
						if ($groupId === $gt['id']) {
							$texts[$gt['lang']] = $gt;
						}
					}

					foreach ($langs as $lang) {
						if (!array_key_exists($lang, $texts)) {
							$isMissingText = true;

							$connection->executeQuery("INSERT INTO dynamicmodule__group_text (lang, group_id, is_published) VALUES (?, ?, ?)", [
								$lang,
								$groupId,
								1,
							]);
						}
					}
				}

				foreach ($members as $m) {
					$memberId = $m['id'];
					$texts    = [];
					foreach ($memberTexts as $mt) {
						if ($memberId === $mt['id']) {
							$texts[$mt['lang']] = $mt;
						}
					}

					foreach ($langs as $lang) {
						if (!array_key_exists($lang, $texts)) {
							$isMissingText = true;

							$seoId = null;
							if (array_key_exists($defaultLocale, $texts) && $m['seo_id']) {
								$connection->prepare("INSERT INTO core__seo (title, description) VALUES ('', '')")
									->executeQuery();
								$seoId = $connection->lastInsertId();
							}

							$stmt = $connection->prepare("INSERT INTO dynamicmodule__member_text (lang, member_id, is_published, seo_id) VALUES (:lang, :memberId, :isPublished, :seoId)");
							$stmt->bindValue('lang', $lang);
							$stmt->bindValue('memberId', $memberId);
							$stmt->bindValue('isPublished', 1);
							$stmt->bindValue('seoId', $seoId);
							$stmt->executeQuery();
						}
					}
				}

				if (!$isMissingText) {
					$output->writeln('All langs already filled');
				}

				if ($isMissingText) {
					$output->writeln('COMPLETED');
				}
			});
		} catch (Exception) {
			$output->writeln('ERROR, PLEAS CHECK LOG');

			return Command::FAILURE;
		}

		return Command::SUCCESS;
	}

}
