<?php declare(strict_types = 1);

namespace DynamicModule\Model\Entities;

use Core\Model\Entities\TIdUnsigned;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Navigations\Model\Entities\Navigation;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'dynamicmodule__virtual_url')]
#[ORM\Entity]
#[ORM\EntityListeners([TranslateListener::class])]
class VirtualUrl
{
	use TIdUnsigned;
	use TTranslateListener;

	#[ORM\Column(type: Types::SMALLINT, options: ['unsigned' => true, 'default' => 0])]
	protected int $isLocked = 0;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $siteIdent;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, options: ['default' => 'CURRENT_TIMESTAMP'])]
	public \DateTime $created;

	#[ORM\Column(type: Types::STRING, length: 50, unique: true, nullable: true)]
	public ?string $relationHash = null;

	#[ORM\ManyToOne(targetEntity: Navigation::class)]
	#[ORM\JoinColumn(name: 'navigation', referencedColumnName: 'id', nullable: false, onDelete: 'CASCADE')]
	public Navigation $navigation;

	/**
	 * @var Collection<Group>
	 */
	#[ORM\JoinTable(name: 'dynamicmodule__virtual_url_group')]
	#[ORM\ManyToMany(targetEntity: Group::class, indexBy: 'id')]
	#[ORM\JoinColumn(name: 'virtual_url_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'feature_group_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Collection $groups;

	/**
	 * @var Collection<FeatureValue>
	 */
	#[ORM\JoinTable(name: 'dynamicmodule__virtual_url_feature_value')]
	#[ORM\ManyToMany(targetEntity: FeatureValue::class, indexBy: 'id')]
	#[ORM\JoinColumn(name: 'virtual_url_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'feature_value_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Collection $features;

	/**
	 * @var Collection<string, VirtualUrlText>
	 */
	#[ORM\OneToMany(targetEntity: VirtualUrlText::class, mappedBy: 'virtualUrl', indexBy: 'locale')]
	public Collection $texts;

	#[ORM\Column(type: Types::SMALLINT, options: ['unsigned' => true, 'default' => 0])]
	public int $addToSitemap = 0;

	#[ORM\Column(type: Types::JSON, nullable: true)]
	private ?array $params = null;

	public function __construct(string $sideIdent, string $relationHash, Navigation $navigation)
	{
		$this->siteIdent    = $sideIdent;
		$this->relationHash = $relationHash;
		$this->navigation   = $navigation;
		$this->texts        = new ArrayCollection();
		$this->groups       = new ArrayCollection();
		$this->features     = new ArrayCollection();
		$this->created      = new DateTime;
	}

	public function getText(string $locale = null): ?VirtualUrlText
	{
		if ($locale) {
			return $this->texts[$locale] ?? null;
		}

		return $this->texts->first() ?: null;
	}

	/** @return VirtualUrlText[] */
	public function getTexts(): array { return $this->texts->toArray(); }

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			if (!is_array($this->params)) {
				$this->params = [];
			}

			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}
}
