<?php declare(strict_types = 1);

namespace DynamicModule\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\Sites;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\FrontModule\Model\FilterUrlHelper;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Gallery\FrontModule\Model\Images;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplatePages;

class Group extends BaseNavigation implements INavigationItem
{
	protected        $title           = 'group';
	protected        $presenter       = 'DynamicModule:Front:Default';
	protected        $action          = 'group';
	protected string $presenterDetail = 'DynamicModule:Front:Default';
	protected string $actionDetail    = 'member';

	protected ?string $navFormSite = null;

	public function __construct(
		protected TemplatePages   $templatePagesService,
		protected Members         $members,
		protected Groups          $groups,
		protected Images          $images,
		protected FilterUrlHelper $filterUrlHelper,
		protected Sites           $sites
	)
	{
	}

	/**
	 * @return string
	 */
	public function getGroup() { return 'Pages'; }

	public function navigationFormCustomHandle(NavigationForm $component, array $data = []): void
	{
		$this->navFormSite = $data['site'];
		$component->loadComponent($data['componentId']);
		$component->redrawControl('component');
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();
		$site      = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages     = $this->templatePagesService->getOptionsForSelectGrouped();

		$container->addSelect('module', 'dynamicModule.navigationForm.module', Helper::getAllModules());
		$container->addSelect('templatePage', 'dynamicModule.navigationForm.templatePage', $pages);
		$container->addBool('allowMemberDetail', 'dynamicModule.navigationForm.allowMemberDetail')
			->setDefaultValue(0);
		$container->addSelect(
			'templatePageMemberDetail',
			'dynamicModule.navigationForm.templatePageMemberDetail',
			['' => ''] + $pages,
		);

		$baseForm->onAnchor[] = function() use ($container, $baseForm) {
			if (isset($baseForm->getValues()['component']['module'])) {
				$qb = $this->groups->getQueryBuilderByModule($baseForm->getValues()['component']['module'])
					->orderBy('g.lft');
				if (!DynamicModuleConfig::load('multiLangPublication')) {
					$qb->andWhere('g.isPublished = 1');
				}
				$groups = [];

				foreach ($qb->getQuery()->getResult() as $g) {
					$groups[] = [
						'id'     => $g->getId(),
						'parent' => $g->getLevel() === 1 ? 0 : ($g->parent ? $g->parent->getId() : 0),
						'name'   => $g->title,
					];
				}

				/** @var NavigationForm $parent */
				$parent = $baseForm->getParent();
				$container->addCheckboxNestedList('baseGroup', 'dynamicModule.navigationForm.baseGroup', $groups)
					->setDefaultValue($parent->navigation->componentParams['baseGroup']);
			}
		};

		return $container;
	}

	/**
	 * @inheritdoc
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return (
			($this->presenter == $presenter && $this->action == $action)
			|| ($this->presenterDetail == $presenter && $this->actionDetail == $action)
		);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed|null
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$urlParams['path'] = NavigationsHelper::removeUrlSuffix($urlParams['path']);

		$params = explode('/', $urlParams['path']);

		if (count($params) > 1) {
			$return = [
				'presenter' => $this->presenterDetail,
				'action'    => $this->actionDetail,
			];

			$checkedMember = $this->members->checkRoute(
				(array) $navigation->componentParams['baseGroup'],
				$navigation->componentParams['module'],
				end($params),
			);

			if (!$checkedMember) {
				$tmp = $params;
				array_shift($tmp);
				while (count($tmp) > 1 && !$checkedMember) {
					$checkedMember = $this->members->checkRoute(
						$navigation->componentParams['baseGroup'],
						$navigation->componentParams['module'],
						implode('/', $tmp),
					);
					array_shift($tmp);
				}
			}

			if ($checkedMember) {
				$return['id'] = (int) $checkedMember['id'];
			} else {
				$id = explode('-', end($params), 2);

				if ($id[0] && $this->members->checkRoute(
						$navigation->componentParams['baseGroup'],
						$navigation->componentParams['module'],
						$id[0],
					)) {
					$return['id'] = (int) $id[0];
				}
			}

			$navigation->componentParams['isMember'] = true;
			$navigation->componentParams['isGroup']  = false;

			if (isset($return['id'])) {
				return $return;
			}
		}

		if ($urlParams['path']) {
			$tmp = explode('/', $urlParams['path']);

			if (end($tmp) !== $navigation->alias) {
				return null;
			}
		}

		$navigation->componentParams['isMember'] = false;
		$navigation->componentParams['isGroup']  = true;

		if (isset($urlParams['filter-filter']) && $urlParams['path'] === ltrim($navigation->link, '/')) {
			$filters = [];
			foreach ($urlParams['filter-filter'] as $v) {
				if (!is_array($v)) {
					$v = explode('|', (string) $v);
				}

				foreach ($v as $vv) {
					$filters[] = (int) $vv;
				}
			}

			$relationData = $this->filterUrlHelper->validateRelationData([
				FilterUrlHelper::$keyNav    => $navigation->getId(),
				FilterUrlHelper::$keyValues => $filters,
			]);

			if ($relationData) {
				$urlExist = $this->filterUrlHelper->findOne($this->sites->getCurrentSite()->getIdent(), $this->translator->getLocale(), $relationData);

				if ($urlExist && $urlExist !== $urlParams['path']) {
					header('Location: /' . ltrim($urlExist, '/'));
				}
			}
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['module'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$endSuffix = NavigationConfig::load('urlSuffix');

		if (isset($urlParams['id']) && is_numeric($urlParams['id'])) {
			if ($urlParams['action'] === $this->actionDetail) {
				$member = $this->members->checkRoute(
					$navigation->componentParams['baseGroup'] ?? [],
					$navigation->componentParams['module'],
					(int) $urlParams['id'],
				);

				if ($member) {
					return '/' . $navigation->alias . '/' . $member['url'] . ($endSuffix ?: '');
				}

				return null;
			}

			if ($urlParams['action'] === $this->action) {
				if (!in_array($urlParams['id'], $navigation->componentParams['baseGroup'] ?: [])) {
					return null;
				}

				$g = $this->groups->checkRoute((int) $urlParams['id'], $navigation->componentParams['module']);
				if ($g) {
					$return = $navigation->link;
				} else {
					return null;
				}
			}
		} else {
			if (isset($urlParams['id']) && $urlParams['id'] !== $navigation->componentParams['module']) {
				return null;
			}

			$return = $navigation->isHomepage ? '/' : '/' . $navigation->alias;
		}

		if (isset($return) && $endSuffix && $navigation->componentParams['allowMemberDetail'] != '1') {
			$return .= $endSuffix;
		}

		return $return ?? null;
	}

	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		if (!$navigation || !$navigation->componentParams['allowMemberDetail']) {
			return $arr;
		}

		$members   = [];
		$galleries = [];
		foreach ($this->members->checkRoute(
			$navigation->componentParams['baseGroup'],
			$navigation->componentParams['module'],
			null,
		) as $id => $row) {
			if (!is_numeric($id)) {
				continue;
			}

			$members[$id] = $row;

			if (isset($row['gallery'])) {
				$galleries[$row['gallery']] = $id;
			}
		}

		if ($galleries) {
			foreach ($this->images->getImagesByAlbum(array_keys($galleries)) as $albumId => $album) {
				$memberId = $galleries[$albumId] ?? null;
				if (!$memberId) {
					continue;
				}

				$members[$memberId]['album'] = $album;
			}
		}

		foreach ($members as $id => $row) {
			$data = [
				'action'    => $this->actionDetail,
				'id'        => $id,
				'presenter' => $this->presenterDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
					$siteMapUrl->setSeo($col, $navigation->getSeo($col));
				}

				if ($row['album']) {
					foreach ($row['album'] as $img) {
						$siteMapUrl->addImage($img);
					}
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$data['activeNavigation'] = $navigation;

			$arr[] = $data;
		}

		return $arr;
	}
}
