<?php declare(strict_types = 1);

namespace DynamicModule\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups as GroupsService;
use DynamicModule\Model\Repository\Members;
use Gallery\FrontModule\Model\Images;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;
use Pages\AdminModule\Model\TemplatePages;

class Groups extends BaseNavigation implements INavigationItem
{
	protected        $title              = 'groups';
	protected        $presenter          = 'DynamicModule:Front:Default';
	protected        $action             = 'groups';
	protected string $presenterDetail    = 'DynamicModule:Front:Default';
	protected string $actionDetail       = 'group';
	protected string $actionMemberDetail = 'member';

	protected ?string $navFormSite = null;

	public function __construct(
		protected TemplatePages  $templatePagesService,
		protected Members        $members,
		protected GroupsService  $groupsService,
		protected Images         $images,
		protected IGroupsFactory $groupsFrontFactory,
	)
	{
	}

	public function getGroup() { return 'Pages'; }

	public function navigationFormCustomHandle(NavigationForm $component, array $data = []): void
	{
		$this->navFormSite = $data['site'];
		$component->loadComponent($data['componentId']);
		$component->redrawControl('component');
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();
		$site      = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages     = $this->templatePagesService->getOptionsForSelectGrouped();

		$container->addSelect('module', 'dynamicModule.navigationForm.module', Helper::getAllModules());
		$container->addSelect('templatePage', 'dynamicModule.navigationForm.templatePage', $pages);
		$container->addSelect('templatePageDetail', 'dynamicModule.navigationForm.templatePageDetail', $pages);
		$container->addSelect(
			'templatePageMemberDetail',
			'dynamicModule.navigationForm.templatePageMemberDetail',
			$pages,
		);

		$container->addBool('allowMemberDetail', 'dynamicModule.navigationForm.allowMemberDetail')
			->setDefaultValue(0);

		return $container;
	}

	/**
	 * @inheritdoc
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return (
			$this->presenter == $presenter && $this->action == $action
			|| $this->presenterDetail == $presenter && ($this->actionDetail == $action || $this->actionMemberDetail == $action)
		);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 */
	public function routerIn($navigation, $urlParams = []): ?array
	{
		$params = explode('/', (string) $urlParams['path']);

		if (count($params) > 1) {
			$return = [
				'presenter' => $this->presenterDetail,
				'action'    => $this->actionDetail,
			];

			$alias        = end($params);
			$checkedGroup = $this->groupsService->checkRoute($alias, $navigation->componentParams['module']);

			if ($checkedGroup) {
				$return['id'] = (int) $checkedGroup['id'];
			} else {
				$id = explode('-', end($params), 2);

				if ($id[0] && $this->members->checkRoute([],
						$navigation->componentParams['module'],
						$id[0]) && $navigation->componentParams['allowMemberDetail'] === '1') {
					$return['id']                            = (int) $id[0];
					$return['action']                        = $this->actionMemberDetail;
					$navigation->componentParams['isMember'] = true;
					$navigation->componentParams['isGroup']  = false;
				} else if ($id[0] && $this->groupsService->checkRoute($id[0], $navigation->componentParams['module'])) {
					$return['id']                            = (int) $id[0];
					$navigation->componentParams['isMember'] = false;
					$navigation->componentParams['isGroup']  = true;
				}
			}

			if (isset($return['id'])) {
				return $return;
			}
		}

		if ($urlParams['path']) {
			$tmp = explode('/', (string) $urlParams['path']);

			if (end($tmp) !== $navigation->alias) {
				return null;
			}
		}

		$navigation->componentParams['isMember'] = false;
		$navigation->componentParams['isGroup']  = true;

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['module'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$endSuffix = NavigationConfig::load('urlSuffix');

		if (isset($urlParams['id']) && is_numeric($urlParams['id'])) {
			if ($urlParams['action'] === $this->actionDetail) {
				$g = $this->groupsService->checkRoute((int) $urlParams['id'], $navigation->componentParams['module']);
				if ($g) {
					return '/' . $navigation->alias . '/' . $g['url'];
				}

				return null;
			}

			if ($urlParams['action'] === $this->actionMemberDetail && $navigation->componentParams['allowMemberDetail'] === '1') {
				$m = $this->members->checkRoute([], $navigation->componentParams['module'], $urlParams['id']);
				if ($m) {
					//					if (isset($member['url']) && Strings::startsWith($member['url'], '/')) {
					//						return $member['url'];
					//					}

					return '/' . $navigation->alias . '/' . $m['url'] . ($endSuffix ?: '');
				}

				return null;
			}

			if ($urlParams['action'] === $this->action) {
				$g = $this->groupsService->checkRoute((int) $urlParams['id'], $navigation->componentParams['module']);
				if ($g) {
					return '/' . $g['url'];
				}

				return null;
			}
		}

		return $navigation->isHomepage ? '/' : '/' . $navigation->alias;
	}

	/**
	 * @inheritdoc
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		if (!$navigation || !$navigation->componentParams['allowMemberDetail']) {
			return $arr;
		}

		$membersSeo = [];
		$groupsRepo = $this->groupsFrontFactory->create($navigation->componentParams['module']);

		foreach ($groupsRepo->getByIds([]) as $row) {
			if ($row->getLvl() <= 0) {
				continue;
			}

			$data = [
				'action'    => $this->actionDetail,
				'id'        => $row->getId(),
				'presenter' => $this->presenterDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
					$siteMapUrl->setSeo($col, $navigation->getSeo($col));
				}

				foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
					if ($row->seo && isset($row->seo->{$col})) {
						$siteMapUrl->setSeo($col, (string) $row->seo->{$col});
					}
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			foreach ($row->members as $member) {
				$membersSeo[$member->getId()] = $member->seo;
			}

			$data['activeNavigation'] = $navigation;

			$arr[] = $data;
		}

		if ($navigation->componentParams['allowMemberDetail'] === '1' && $navigation->componentParams['module']) {
			$members   = [];
			$galleries = [];
			foreach ($this->members->checkRoute(null, $navigation->componentParams['module']) as $id => $row) {
				if (!is_numeric($id)) {
					continue;
				}

				$members[$id] = $row;

				if (isset($row['gallery'])) {
					$galleries[$row['gallery']] = $id;
				}
			}

			if ($galleries) {
				foreach ($this->images->getImagesByAlbum(array_keys($galleries)) as $albumId => $album) {
					$memberId = $galleries[$albumId] ?? null;
					if (!$memberId) {
						continue;
					}

					$members[$memberId]['album'] = $album;
				}
			}

			foreach ($members as $id => $row) {
				$data = [
					'action'    => $this->actionMemberDetail,
					'id'        => $id,
					'presenter' => $this->presenterDetail,
				];

				if ($siteMapData) {
					$siteMapUrl = new SiteMapUrl();
					foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
						$siteMapUrl->setSeo($col, $navigation->getSeo($col));
					}

					$memberSeo = $membersSeo[$row['id']] ?? null;
					if ($memberSeo) {
						foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority'] as $col) {
							if (isset($memberSeo->{$col})) {
								$siteMapUrl->setSeo($col, (string) $memberSeo->{$col});
							}
						}
					}

					if ($row['album']) {
						foreach ($row['album'] as $img) {
							$siteMapUrl->addImage($img);
						}
					}

					$data['siteMapData']['default'] = $siteMapUrl;
				}

				$data['activeNavigation'] = $navigation;

				$arr[] = $data;
			}
		}

		$membersSeo = null;

		return $arr;
	}
}
