<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\ArrayCollection;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\Model\CacheService;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Feature;
use DynamicModule\Model\Entities\FeatureText;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FeatureForm extends BaseControl
{
	public ?Feature        $feature = null;
	protected Features     $featureServices;
	protected CacheService $cacheService;

	public function __construct(
		Features     $features,
		CacheService $cacheService
	)
	{
		$this->featureServices = $features;
		$this->cacheService    = $cacheService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'dynamicModule.feature.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addBool('isPublished', 'dynamicModule.feature.isPublished')
			->setDefaultValue(1);

		if (DynamicModuleConfig::load('allowVirtualUrls')) {
			$form->addSelect('urlPosition', 'dynamicModule.feature.urlPosition', [
				0 => 'dynamicModule.feature.beforeTitle',
				1 => 'dynamicModule.feature.afterTitle',
			]);
			$form->addText('urlPositionPriority', 'dynamicModule.feature.urlPositionPriority')
				->setHtmlAttribute('type', 'number')
				->setDefaultValue(0);

			$form->addText('nameMulti', 'dynamicModule.feature.nameMulti')
				->setIsMultilanguage();
			$form->addSelect('urlPositionMulti', 'dynamicModule.feature.urlPositionMulti', [
				0 => 'dynamicModule.feature.beforeTitle',
				1 => 'dynamicModule.feature.afterTitle',
			]);
			$form->addText('urlPositionMultiPriority', 'dynamicModule.feature.urlPositionMultiPriority')
				->setHtmlAttribute('type', 'number')
				->setDefaultValue(0);
		}

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$feature = $this->feature ?: new Feature();

			$feature->isPublished = (int) $values->isPublished;

			if (DynamicModuleConfig::load('allowVirtualUrls')) {
				$feature->urlPosition              = (int) $values->urlPosition;
				$feature->urlPositionMulti         = (int) $values->urlPositionMulti;
				$feature->urlPositionPriority      = (int) $values->urlPositionPriority;
				$feature->urlPositionMultiPriority = (int) $values->urlPositionMultiPriority;
			}

			$this->em->persist($feature);

			$langValues = $form->convertMultilangValuesToArray();
			$texts      = $feature->texts->toArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] === '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}

					continue;
				}

				$text       = $texts[$lang] ?? new FeatureText($feature, $lang);
				$text->name = $vals['name'];

				if (DynamicModuleConfig::load('allowVirtualUrls')) {
					$text->nameMulti = $vals['nameMulti'];
				}

				$this->em->persist($text);
				$texts[$lang] = $text;
			}

			$feature->texts = new ArrayCollection($texts);

			$this->em->flush();
			$this->em->commit();

			$form->addCustomData('featureId', $feature->getId());
			$this->presenter->flashMessageSuccess('default.saved');

			$this->cacheService->removeFeature($feature->getId());

			return true;
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setFeature(int $id): void
	{
		$this->feature = $this->featureServices->get($id);

		if ($this->feature) {
			$form = $this['form'];

			$form->setDefaults([
				'isPublished'              => $this->feature->isPublished,
				'urlPosition'              => $this->feature->urlPosition,
				'urlPositionMulti'         => $this->feature->urlPositionMulti,
				'urlPositionPriority'      => $this->feature->urlPositionPriority,
				'urlPositionMultiPriority' => $this->feature->urlPositionMultiPriority,
			]);

			$mTexts = $form->prepareMultilangTextsForForm([
				'name',
				'nameMulti',
			], $this->feature->texts->toArray());
			$form->setDefaults($mTexts['texts']);
		} else {
			throw new InvalidArgumentException;
		}
	}

}
