<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\AdminModule\Model\FeatureValues;
use DynamicModule\Model\Entities\Feature;
use DynamicModule\Model\Entities\FeatureValue;
use Nette\Application\AbortException;
use Nette\Utils\Html;

class FeatureValuesGrid extends BaseControl
{
	protected Feature       $feature;
	protected Features      $featuresService;
	protected FeatureValues $featureValuesServices;

	public function __construct(
		Feature       $feature,
		Features      $features,
		FeatureValues $featureValues
	)
	{
		$this->feature               = $feature;
		$this->featuresService       = $features;
		$this->featureValuesServices = $featureValues;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param string|int $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->featureValuesServices->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->featureValuesServices->getEr()->createQueryBuilder('fv')->addSelect('fvt')
			->join('fv.texts', 'fvt', Join::WITH, 'fvt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->andWhere('fv.feature = :id')->setParameter('id', $this->feature->getId())
			->orderBy('fv.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('featureValuesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.value')->setRenderer(function(FeatureValue $row) {
			return Html::el('a', [
				'href' => $this->presenter->link('ModuleFeatures:editFeatureValue', $row->getId()),
			])
				->setText($row->texts->get($this->translator->getLocale())->name);
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'default.unPublish')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filters
		$grid->addFilterText('name', '', 'fvt.name');
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Prototype
		$grid->getColumn('isPublished')->getElementPrototype('td')->addClass('w1nw');

		// Actions
		$grid->addAction('edit', '', 'ModuleFeatures:editFeatureValue')
			->setIcon('edit')
			->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(string $id, string $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublish((int) $id, (int) $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		$this['grid']->redrawItem((int) $id);
		$presenter->redrawControl('flashes');
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureValuesServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}
}
