<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\Features;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\Model\Entities\Feature;
use Nette\Application\AbortException;
use Nette\Utils\Html;

class FeaturesGrid extends BaseControl
{
	protected Features $featureServices;

	public function __construct(
		Features $features
	)
	{
		$this->featureServices = $features;
	}

	public function render(): void
	{
		$this->template->render(__DIR__ . '/FeaturesGrid.latte');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */


	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->featureServices->getEr()->createQueryBuilder('f')
			->addSelect('ft')
			->join('f.texts', 'ft', Join::WITH, 'ft.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('f.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('featuresGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function(Feature $row) {
			$texts = $row->texts->get($this->translator->getLocale());

			return $texts
				? Html::el()->addHtml(Html::el('a', ['href' => $this->presenter->link('ModuleFeatures:editFeature', $row->getId())])
					->setText($texts->name))
				: '';
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'default.unPublish')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '', ['ft.name']);
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Actions
		$grid->addAction('edit', '', 'ModuleFeatures:editFeature')
			->setIcon('edit')
			->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Prototype
		$grid->getColumn('isPublished')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->featureServices->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureServices->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$presenter->redrawControl('flashes');
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}
}
