<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\Seo;
use Core\Model\TemplateReader\TemplateReader;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\AdminModule\Model\Events\GroupEvent;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Feature;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupFeature;
use DynamicModule\Model\Entities\GroupText;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Tracy\Debugger;

/**
 * @property Group|null $entity
 */
class GroupForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int            $groupId = null;
	protected string       $moduleKey;
	protected Groups       $groups;
	protected ?Group       $group   = null;
	protected SeoContainer $seoContainer;
	protected Features     $features;

	public function __construct(
		string       $moduleKey,
		Groups       $groups,
		SeoContainer $seoContainer,
		Features     $features
	)
	{
		$this->moduleKey    = $moduleKey;
		$this->groups       = $groups;
		$this->seoContainer = $seoContainer;
		$this->features     = $features;
	}

	/**
	 * TODO pokud bylo nějaké tt typu image, tak pri ulozeni
	 * formu a nacteni jine polozky zustaly v tt image stejne data.
	 *
	 * @param IComponent $presenter
	 */
	protected function attached($presenter): void
	{
		if ($this->groupId) {
			$this->entity = $this->groups->get($this->groupId);
			$isGet        = $this->httpRequest->isMethod('GET');
			$q            = $this->httpRequest->getQuery('do');
			$postDo       = $this->httpRequest->getPost('_do');
			if (
				(!$isGet && $postDo && Strings::contains($postDo, 'uploadFromServerForm'))
				|| (!$isGet && $postDo && Strings::contains($postDo, 'groupForm-form-submit'))
				|| ($isGet && $q && Strings::endsWith($q, 'showGallery'))
				|| (!$isGet && $q && Strings::endsWith($q, 'updateImage'))
				|| (!$isGet && $q && Strings::contains($q, 'galleryGallery'))
				|| (!$isGet && $q && (Strings::endsWith($q, 'Gallery-onEmpty')
						|| Strings::endsWith($q, 'Gallery-upload')
						|| Strings::endsWith($q, 'Gallery-removeImage')))
			) {
				$this->templateReader->loadTemplateComponents($this['form']->getComponent('component'), $this->entity->template, $this->entity ? [$this->entity] : []);
			}
		}
		parent::attached($presenter);
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Components
	 */

	public function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateGroup', TEMPLATES_DIR, ucfirst($this->moduleKey)));
		$this->templateReader->setTranslateKey(sprintf('%s.templateGroup', $this->moduleKey));

		$templates = $this->templateReader->getTemplates();
		$form      = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);
		if (DynamicModuleConfig::load('multiLangPublication')) {
			$form->getComponent('isPublished')->setIsMultilanguage();
		}
		$form->addSelect('parent', 'default.parent', $this->getPossibleParents())
			->setTranslator(null);
		$form->addHidden('id', $this->group ? $this->group->getId() : null);
		$form->addSelect('template', $this->t('default.templates'), $templates)
			->setPrompt($this->t('default.choose'))
			->setRequired();
		$form->addComponent(new BaseContainer, 'component');

		if (DynamicModuleConfig::load('allowVirtualUrls')) {
			$form->addSelect('rod', 'dynamicModule.group.rod', [
				''  => '',
				'm' => 'dynamicModule.rodList.m',
				'z' => 'dynamicModule.rodList.z',
				's' => 'dynamicModule.rodList.s',
			]);
		}

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$form->addComponent($this->seoContainer->getContainer(true), 'seo');
		}

		if (DynamicModuleConfig::load('allowFeatures')) {
			$filters = [];
			foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
				         ->join('f.texts', 'ft', Join::WITH, 'ft.lang = :lang')
				         ->setParameters([
					         'lang' => $this->translator->getLocale(),
				         ])
				         ->orderBy('f.position')
				         ->getQuery()->getArrayResult() as $row) {
				$filters[$row['id']] = $row['name'];
			}

			$form->addSortableCheckboxList('filters', $this->t('dynamicModule.group.filters'), $filters)
				->setTranslator(null);
		}

		$form->addSaveCancelControl()
			->closeModalOnCancel();

		$form->onValidate[] = function(BaseForm $form, ArrayHash $values) {
			$this->redrawControl('formErrors');
		};
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var GroupText[] $texts */
			$texts = [];

			if (!Validators::isNone($values->id)) {
				$this->group = $this->groups->get((int) $values->id);
			}

			if (!$this->group) {
				$this->group = new Group($this->moduleKey, $values->title);
			} else {
				$texts = $this->group->texts->toArray();
			}

			$this->group->title    = $values->title;
			$this->group->template = $values->template;

			if (DynamicModuleConfig::load('allowVirtualUrls')) {
				$this->group->rod = $values->rod;
			}

			if (DynamicModuleConfig::load('multiLangPublication')) {
				foreach ($langValues as $l => $v) {
					if (!isset($texts[$l])) {
						$texts[$l] = new GroupText($this->group, $l);
					}

					$texts[$l]->isPublished = (int) $v['isPublished'];

					$this->em->persist($texts[$l]);
				}
				$this->group->setEntityText($texts);

				if (count($texts) === 1) {
					$k                      = array_key_first($texts);
					$texts[$k]->isPublished = (int) $values->isPublished[$k];
				}
			} else {
				$this->group->isPublished = $values->isPublished;
			}

			if ($values->parent == '') {
				$this->group->setParent($this->groups->getRootGroup($this->moduleKey));
			} else {
				$this->group->setParent($this->groups->get($values->parent));
			}

			// FILTRY
			if (DynamicModuleConfig::load('allowFeatures')) {
				$formFilters  = $values->filters;
				$groupFilters = $this->group->filters->toArray();

				// nové
				foreach (array_diff($formFilters, array_keys($groupFilters)) as $k => $v) {
					$groupFilter = new GroupFeature($this->group, $this->features->getReference($v), $k);
					$this->em->persist($groupFilter);
					$this->group->filters->add($groupFilter);
				}

				// odstranit
				foreach (array_diff(array_keys($groupFilters), $formFilters) as $v) {
					$groupFilter = $this->group->filters->get($v);
					if ($groupFilter) {
						$this->group->filters->remove($v);
						$this->em->remove($groupFilter);
					}
				}

				// seřadit
				foreach (array_values($formFilters) as $k => $v) {
					$groupFilter = $this->group->filters->get($v['id']);
					if ($groupFilter) {
						$groupFilter->position = $k;
						$this->em->persist($groupFilter);
					}
				}
			}

			// Nacteni komponent ze sablony
			$this->templateReader->loadTemplateComponents($this['form']->getComponent('component'), $this->entity->template ?: $values->template, $this->entity ? [$this->entity] : []);

			$data = Helper::transformTemplateArrayData($form->getHttpData()['component'] ?: [], $form->getComponents(true));
			$this->group->setTexts($data);

			if (DynamicModuleConfig::load('member.allowSeo')) {
				foreach ($langValues as $lang => $vals) {
					if (!isset($texts[$lang])) {
						$texts[$lang] = new GroupText($this->group, $lang);
					}

					$seo                    = $texts[$lang]->seo ?: new Seo();
					$seo->title             = $vals['seo']['title'];
					$seo->description       = $vals['seo']['description'];
					$seo->robots            = $vals['seo']['robots'] ?? 'index, follow';
					$seo->canonical         = $vals['seo']['canonical'] ?? 'self';
					$seo->addToSiteMap      = (int) ($vals['seo']['addToSiteMap'] ?? 0);
					$seo->siteMapChangeFreq = $vals['seo']['siteMapChangeFreq'] ?? 'monthly';
					$seo->siteMapPriority   = $vals['seo']['siteMapPriority'] ?? '0.5';

					$this->em->persist($seo);
					$texts[$lang]->seo = $seo;
					$this->em->persist($texts[$lang]);
				}

				$this->group->setEntityText($texts);
			}

			$this->em->persist($this->group);
			$this->em->flush();

			$this->eventDispatcher->dispatch(new GroupEvent($this->group), 'dynamicModule.admin.' . $this->group->moduleKey . '.groupChanged');

			$form->addCustomData('groupId', $this->group->getId());
			$this->presenter->flashMessageSuccess('default.save');

			$cache = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);
		} catch (\Exception $e) {
			Debugger::log($e);
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	public function setGroup(Group $group): void
	{
		$this->groupId = $group->getId();
		$this->group   = $group;

		$form = $this['form'];
		$form->setSubmittedBy(null);
		$form->setDefaults([
			'title'       => $group->title,
			'isPublished' => $group->isPublished,
			'id'          => $group->getId(),
			'rod'         => $group->rod,
		]);
		$form->getComponent('id')->setValue($group->getId());
		$form->getComponent('parent')->setItems($this->getPossibleParents());

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$d = [];
			foreach ($this->group->getEntityTexts() as $l => $text) {
				$d['isPublished'][$l] = $text->isPublished;
			}

			if ($d) {
				$form->setDefaults($d);
			}
		}


		// FILTRY
		if (DynamicModuleConfig::load('allowFeatures')) {
			$d = [];
			foreach ($this->group->filters->toArray() as $k => $v) {
				/** @var GroupFeature $v */
				if (array_key_exists($k, $form->getComponent('filters')->getItems())) {
					$d[] = $k;
				}
			}
			$form->getComponent('filters')->setDefaultValue($d);
		}

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$this->seoContainer->setDefaultsFromEntity($form->getComponent('seo'), $this->group->getEntityTexts());
		}

		if ($this->group->parent && array_key_exists($this->group->parent->getId(), $form->getComponent('parent')->getItems())) {
			$form->getComponent('parent')->setDefaultValue($this->group->parent->getId());
		}

		if ($this->group->template && array_key_exists($this->group->template, $form->getComponent('template')->getItems())) {
			$form->getComponent('template')->setDefaultValue($this->group->template);

			if ($this->httpRequest->getQuery('do') !== 'groupForm-loadInputs') {
				/** @var BaseContainer $container */
				$container            = $form['component'];
				TemplateReader::$mode = 'dynamicModule';
				$this->templateReader->loadTemplateComponents($container, $this->httpRequest->getPost('template') ?: $this->group->template, $this->entity ? [$this->entity] : []);
				$this->templateReader->setDefaults($container, $this->group, $this->group->template);
			}
		}
	}

	protected function getPossibleParents(): array
	{
		$groups = [null => ''];
		foreach ($this->groups->getQueryBuilderByModule($this->moduleKey)
			         ->orderBy('g.lft')
			         ->getQuery()->getResult() as $g) {
			/** @var Group $g */
			if ($this->group && $this->group->getId() == $g->getId()) {
				continue;
			}

			$groups[$g->getId()] = trim((string) (str_repeat('---', $g->getLevel() - 1) . ' ' . $g->title));
		}

		return $groups;
	}

}
