<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\Seo;
use Core\Model\TemplateReader\TemplateReader;
use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\AdminModule\Model\Features;
use DynamicModule\AdminModule\Model\FeatureValues;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\FeatureValue;
use DynamicModule\Model\Entities\FeatureValueText;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupFeature;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Entities\MemberText;
use DynamicModule\Model\Helper;
use DynamicModule\Model\Repository\Groups;
use DynamicModule\Model\Repository\Members;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Tracy\Debugger;

/**
 * @property Member|null $entity
 */
class MemberForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int       $memberId = null;
	protected string  $moduleKey;
	protected Members $members;
	protected Groups  $groups;
	protected ?Member $member   = null;

	protected SeoContainer  $seoContainer;
	protected Features      $features;
	protected FeatureValues $featureValues;

	private ?array $cFeatures      = null;
	private ?array $cFeatureValues = null;

	public function __construct(
		string        $moduleKey,
		Members       $members,
		Groups        $groups,
		SeoContainer  $seoContainer,
		Features      $features,
		FeatureValues $featureValues
	)
	{
		$this->moduleKey     = $moduleKey;
		$this->members       = $members;
		$this->groups        = $groups;
		$this->seoContainer  = $seoContainer;
		$this->features      = $features;
		$this->featureValues = $featureValues;
	}

	/**
	 * TODO pokud bylo nějaké tt typu image, tak pri ulozeni
	 * formu a nacteni jine polozky zustaly v tt image stejne data.
	 *
	 * @param IComponent $presenter
	 */
	protected function attached($presenter): void
	{
		if ($this->memberId) {
			$this->entity = $this->members->get($this->memberId);
			$isGet        = $this->httpRequest->isMethod('GET');
			$q            = $this->httpRequest->getQuery('do');
			$postDo       = $this->httpRequest->getPost('_do');
			if (
				(!$isGet && $postDo && Strings::contains($postDo, 'uploadFromServerForm'))
				|| ($isGet && $q && Strings::endsWith($q, 'showGallery'))
				|| (!$isGet && $q && Strings::endsWith($q, 'updateImage'))
				|| (!$isGet && $q && Strings::contains($q, 'galleryGallery'))
				|| (!$isGet && $q && (Strings::endsWith($q, 'Gallery-onEmpty')
						|| Strings::endsWith($q, 'Gallery-upload')
						|| Strings::endsWith($q, 'Gallery-removeImage')))
			) {
				$this->templateReader->loadTemplateComponents($this['form']->getComponent('component'), $this->entity->template, $this->entity ? [$this->entity] : []);
			}
		}
		parent::attached($presenter);
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		if (DynamicModuleConfig::load('allowFeatures')) {
			$features                      = $this->getFeatures();
			$this->template->features      = $features;
			$this->template->featureValues = $this->getFeatureValues();

			$defaults = [];

			if ($this->member) {
				foreach ($this->member->groups as $group) {
					foreach ($group->group->filters as $row) {
						/** @var GroupFeature $row */
						$defaults[$row->feature->getId()] = [
							'feature' => $row->feature->getId(),
							'values'  => [],
						];
					}
				}

				foreach ($this->member->features as $row) {
					if (isset($defaults[$row->feature->getId()])) {
						$defaults[$row->feature->getId()]['values'][] = $row->getId();
					}
				}
			}

			$this->template->defaultFeatures = $defaults;

			$this->template->featuresJson = Json::encode([
				'features' => $features,
				'values'   => $this->getFeatureValues(),
			]);
		}

		$this->template->render($this->getTemplateFile());
	}

	public function handleAddFeatureValue(string $id, string $val): void
	{
		if (!$val) {
			$this->presenter->sendJson([]);
		}

		$exist = $this->em->getRepository(FeatureValueText::class)->createQueryBuilder('fvt')
			->select('IDENTITY(fvt.featureValue) as id, fvt.name as name')
			->innerJoin('fvt.featureValue', 'fv', Join::WITH, 'fv.feature = :feature')
			->where('fvt.name = :v')
			->setParameters([
				'feature' => $id,
				'v'       => $val,
			])
			->setMaxResults(1)
			->getQuery()->getResult()[0] ?? null;

		if (!$exist) {
			$featureValue              = new FeatureValue();
			$featureValue->feature     = $this->features->getReference((int) $id);
			$featureValue->isPublished = 1;
			$this->em->persist($featureValue);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$featureValueText       = new FeatureValueText($featureValue, $lang->getTag());
				$featureValueText->name = $val;
				$this->em->persist($featureValueText);
			}

			$this->em->flush();
			$exist = [
				'id'      => $featureValue->getId(),
				'name'    => $val,
				'created' => true,
			];
		}

		$this->presenter->sendJson([
			'data' => $exist,
		]);
	}

	public function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateMember', TEMPLATES_DIR, ucfirst($this->moduleKey)));
		$this->templateReader->setTranslateKey(sprintf('%s.templateMember', $this->moduleKey));

		$templates = $this->templateReader->getTemplates();

		$groups = [];
		foreach ($this->groups->getQueryBuilderByModule($this->moduleKey)
			         ->orderBy('g.lft')
			         ->getQuery()->getResult() as $g) {
			$groups[$g->getId()] = trim((string) (str_repeat('---', $g->getLevel() - 1) . ' ' . $g->title));
		}

		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);
		if (DynamicModuleConfig::load('multiLangPublication')) {
			$form->getComponent('isPublished')->setIsMultilanguage();
		}
		$form->addCheckboxList('group', 'default.group', $groups)
			->setTranslator(null)
			->setRequired();
		$form->addSelect('template', $this->t('default.templates'), $templates)
			->setPrompt($this->t('default.choose'))
			->setRequired();
		$form->addComponent(new BaseContainer, 'component');
		$form->addHidden('id', $this->member ? $this->member->getId() : null);

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$form->addComponent($this->seoContainer->getContainer(true), 'seo');
		}

		//Features - Vlastnosti
		if (DynamicModuleConfig::load('allowFeatures')) {
			$form->addContainer('features');
		}

		$form->addSaveCancelControl()
			->closeModalOnCancel();

		$form->onValidate[] = function() {
			$this->redrawControl('formErrors');
		};
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var MemberText[] $texts */
			$texts = [];

			if (!Validators::isNone($values->id)) {
				$this->member = $this->members->get((int) $values->id);
			}

			if (!$this->member) {
				$this->member = new Member($values->title, $values->template);
			} else {
				$texts = $this->member->texts->toArray();
			}

			$defaultGroupIds = array_map(static function(Group $group): int {
				return $group->getId();
			}, $this->member->getGroups());

			// add to group
			foreach (array_diff($values->group, $defaultGroupIds) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->addGroup($group);
				}
			}

			// remove from group
			foreach (array_diff($defaultGroupIds, $values->group) as $groupId) {
				$group = $this->groups->get((int) $groupId);

				if ($group) {
					$this->member->removeGroup($group);
				}
			}

			$this->member->title    = $values->title;
			$this->member->template = $values->template;

			if (DynamicModuleConfig::load('multiLangPublication')) {
				foreach (array_keys($this->langsService->getLangs(false)) as $l) {
					if (!isset($langValues[$l]['isPublished'])) {
						$langValues[$l]['isPublished'] = 0;
					}
				}

				foreach ($langValues as $l => $v) {
					if (!isset($texts[$l])) {
						$texts[$l] = new MemberText($this->member, $l);
					}

					$texts[$l]->isPublished = (int) $v['isPublished'];

					$this->em->persist($texts[$l]);
				}
				$this->member->setEntityText($texts);

				if (count($texts) === 1) {
					$k                      = array_key_first($texts);
					$texts[$k]->isPublished = (int) $values->isPublished[$k];
				}
			} else {
				$this->member->isPublished = $values->isPublished;
			}

			//Vlastnosti
			if (DynamicModuleConfig::load('allowFeatures')) {
				$currentFeatures = [];
				$formFeatures    = [];

				foreach ($this->member->features as $fp) {
					$currentFeatures[$fp->getId()] = $fp;
				}

				foreach ($form->getHttpData()['features'] as $v) {
					if ($v) {
						$formFeatures[$v] = $v;
					}
				}

				// Nove
				foreach (array_diff_key($formFeatures, $currentFeatures) as $id) {
					/** @var FeatureValue $fv */
					$fv = $this->em->getReference(FeatureValue::class, $id);

					$this->member->features->set($id, $fv);
				}

				// Stare
				foreach (array_diff_key($currentFeatures, $formFeatures) as $id => $v) {
					$this->member->features->remove($id);
				}


				$this->em->persist($this->member);
			}

			// Nacteni komponent ze sablony
			$this->templateReader->loadTemplateComponents($this['form']->getComponent('component'), $this->entity->template ?: $values->template, $this->entity ? [$this->entity] : []);

			$data = Helper::transformTemplateArrayData($form->getHttpData()['component'] ?: [], $form->getComponents(true));
			$this->member->setTexts($data);

			if (DynamicModuleConfig::load('member.allowSeo')) {
				foreach ($langValues as $lang => $vals) {
					if (!isset($texts[$lang])) {
						$texts[$lang] = new MemberText($this->member, $lang);
					}

					$seo                    = $texts[$lang]->seo ?: new Seo();
					$seo->title             = $vals['seo']['title'];
					$seo->description       = $vals['seo']['description'];
					$seo->robots            = $vals['seo']['robots'] ?? 'index, follow';
					$seo->canonical         = $vals['seo']['canonical'] ?? 'self';
					$seo->addToSiteMap      = (int) ($vals['seo']['addToSiteMap'] ?? 0);
					$seo->siteMapChangeFreq = $vals['seo']['siteMapChangeFreq'] ?? 'monthly';
					$seo->siteMapPriority   = $vals['seo']['siteMapPriority'] ?? '0.5';

					$this->em->persist($seo);
					$texts[$lang]->seo = $seo;
					$this->em->persist($texts[$lang]);
				}

				$this->member->setEntityText($texts);
			}

			$this->em->persist($this->member);
			$this->em->flush();

			$this->eventDispatcher->dispatch(new MemberEvent($this->member), 'dynamicModule.admin.' . $this->member->groups->first()->group->moduleKey . '.memberChanged');

			$form->addCustomData('memberId', $this->member->getId());
			$this->memberId = null;
			$this->presenter->flashMessageSuccess('default.save');

			$cache       = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
			$memberCache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);

			$groups = array_merge($defaultGroupIds, $values->group);
			$groups = array_unique($groups);

			foreach ($this->langsService->getLangs(false) as $lang) {
				foreach ($groups as $groupId) {
					$cache->remove($this->moduleKey . '_' . md5((string) $groupId) . '_' . $lang->getTag());
				}

				$memberCache->remove('dynamicModuleMember/' . $lang->getTag() . '/' . $this->member->getId());
			}
			$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => Members::CACHE_NAMESPACE]);
		} catch (\Exception $e) {
			Debugger::log($e);
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	public function setMember(Member $member): void
	{
		$this->memberId = $member->getId();
		$this->member   = $member;

		$form = $this['form'];
		$form->setSubmittedBy(null);
		$form->setDefaults([
			'title'       => $member->title,
			'isPublished' => $member->isPublished,
			'template'    => $member->template,
			'id'          => $member->getId(),
			'group'       => array_map(static function(Group $group): int {
				return $group->getId();
			}, $member->getGroups()),
		]);
		$form->getComponent('id')->setValue($member->getId());

		if (DynamicModuleConfig::load('multiLangPublication')) {
			$d = [];
			foreach ($this->member->getEntityTexts() as $l => $text) {
				$d['isPublished'][$l] = $text->isPublished;
			}

			if ($d) {
				$form->setDefaults($d);
			}
		}

		if (DynamicModuleConfig::load('member.allowSeo')) {
			$this->seoContainer->setDefaultsFromEntity($form->getComponent('seo'), $member->getEntityTexts());
		}

		if ($this->member->template && array_key_exists($this->member->template, $form->getComponent('template')->getItems())) {
			$form->getComponent('template')->setDefaultValue($this->member->template);

			if ($this->httpRequest->getQuery('do') !== 'memberForm-loadInputs') {
				/** @var BaseContainer $container */
				$container            = $form['component'];
				TemplateReader::$mode = 'dynamicModule';
				$this->templateReader->loadTemplateComponents($container, $this->httpRequest->getPost('template') ?: $this->member->template, [$this->member]);
				$this->templateReader->setDefaults($container, $this->member, $this->member->template);
			}
		}
	}

	protected function getFeatures(): array
	{
		if ($this->cFeatures === null) {
			$this->cFeatures = [];
			foreach ($this->features->getEr()
				         ->createQueryBuilder('f')
				         ->select('f.id, ft.name')
				         ->join('f.texts', 'ft', Join::WITH, 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('f.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$v['name'] = trim((string) $v['name']);

				$this->cFeatures[$v['id']] = $v;
			}
		}

		return $this->cFeatures;
	}

	protected function getFeatureValues(?int $id = null): array
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues = [];

			foreach ($this->featureValues->getEr()
				         ->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.texts', 'fvt', Join::WITH, 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('fv.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$this->cFeatureValues[$v['featureId']][] = [
					'id'   => $v['id'],
					'name' => trim($v['name']),
				];
			}
		}

		return $id ? $this->cFeatureValues[$id] ?? [] : $this->cFeatureValues;
	}
}
