<?php declare(strict_types = 1);

namespace DynamicModule\AdminModule\Components\VirtualUrl;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DynamicModule\AdminModule\Model\VirtualUrls;
use DynamicModule\FrontModule\Model\FilterUrlHelper;
use DynamicModule\Model\CacheService;
use DynamicModule\Model\Entities\VirtualUrl;
use DynamicModule\Model\Entities\VirtualUrlText;
use Exception;
use Nette\Application\UI\InvalidLinkException;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Pages\AdminModule\Model\TemplatePages;

class VirtualUrlTextForm extends TemplateReaderControl
{
	/** @var int|null @persistent */
	public ?int $id = null;

	/**
	 * @var VirtualUrl|null
	 * @phpstan-ignore-next-line
	 */
	protected $entity;

	protected VirtualUrls     $virtualUrls;
	protected CacheService    $cacheService;
	protected TemplatePages   $templatePages;
	protected FilterUrlHelper $filterUrlHelper;

	public function __construct(
		VirtualUrls     $virtualUrls,
		CacheService    $cacheService,
		TemplatePages   $templatePages,
		FilterUrlHelper $filterUrlHelper
	)
	{
		$this->virtualUrls     = $virtualUrls;
		$this->cacheService    = $cacheService;
		$this->templatePages   = $templatePages;
		$this->filterUrlHelper = $filterUrlHelper;
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];
		$this->template->virtualUrl         = $this->entity;

		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param IComponent $presenter
	 *
	 * @throws InvalidLinkException
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$httpRequest = $this->presenter->getHttpRequest();

		$this->setFields();
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var VirtualUrlText[] $texts */
			$texts           = [];
			$groupsCacheKeys = [];
			$componentValues = $form->getComponent('component')->getValues(true);

			if ($this->getVirtualUrl()) {
				$entity = $this->entity;
				$texts  = $entity->texts->toArray();
			} else {
				throw new Exception('Virtual url not found');
			}

			if (!$langValues) {
				foreach ($this->langsService->getLangs(false) as $l => $v) {
					$langValues[$l]['component'] = $componentValues;
				}
			}

			foreach ($langValues as $l => $v) {
				$v = $v['component'];
				if (!isset($texts[$l])) {
					continue;
				}

				foreach ($v as $c => $cValue) {
					/** @phpstan-ignore-next-line */
					$comp = $form['component'][$c];
					if ($comp instanceof BaseContainer === false) {
						continue;
					}

					$tmp = Json::decode(Json::encode($comp->getValues()), Json::FORCE_ARRAY);

					$v[$c] = array_merge($tmp, $cValue);

					if (array_key_exists('types', $tmp) && array_key_exists('types', $cValue)) {
						$v[$c]['types'] = array_merge($tmp['types'], $cValue['types']);
					}
				}

				foreach (array_diff(array_keys($componentValues), array_keys($v)) as $name) {
					if (is_array($componentValues[$name]) && array_key_exists($l, $componentValues[$name])) {
						$v[$name] = $componentValues[$name][$l];
					} else {
						$v[$name] = $componentValues[$name];
					}
				}

				$texts[$l]->setTexts($v);
				$this->em->persist($texts[$l]);
			}

			$flashMessage = $entity->getId() ? 'pages.templateForm.edited' : 'pages.templateForm.added';

			$this->em->flush();

			foreach ($texts as $v) {
				$this->cacheService->filterCache->remove($this->filterUrlHelper->getCacheKeyByUrl($entity->siteIdent, $v->locale, $v->url));
			}

			$this->cacheService->removeMember($entity->getId());

			$form->addCustomData('virtualUrlId', $entity->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function getVirtualUrl(): ?VirtualUrl
	{
		if (!$this->entity && $this->id) {
			$this->entity = $this->virtualUrls->get($this->id);
		}

		return $this->entity;
	}

	public function setVirtualUrl(int $id): void
	{
		$this->entity = $this->virtualUrls->get($id);

		if (!$this->entity) {
			$this->presenter->error();
		}

		$this->id = $id;

		$form = $this['form'];
		$d    = [];
	}

	protected function setFields(): void
	{
		$parsedTemplate = null;

		$navigation     = $this->getVirtualUrl()->navigation;
		$templatePageId = $navigation->componentParams['templatePage'] ?? null;
		if (!$templatePageId) {
			return;
		}

		$templatePage = $this->templatePages->get($templatePageId);
		if (!$templatePage) {
			return;
		}

		$parsedTemplate = $this->parseTemplate($templatePage->template);

		if ($parsedTemplate['template'] && $this->httpRequest->getQuery('do') != 'virtualUrlTextForm-loadInputs') {
			$tmp = explode('|', $parsedTemplate['template']);
			if (count($tmp) === 2) {
				$this->templateReader->setTemplatesDir($this->pathsService->getTemplatePagesDir($tmp[0]));
			}

			/** @var BaseContainer $container */
			$container = $this['form']['component'];
			$this->templateReader->loadTemplateComponents($container, $parsedTemplate['file'], $this->getVirtualUrl()->texts->toArray());
			$this->templateReader->setDefaults($container, $this->getVirtualUrl()->texts->toArray(), $parsedTemplate['file']);
		}
	}

	protected function parseTemplate(string $template): array
	{
		$result = [
			'template' => $template,
			'file'     => $template,
		];

		if (strpos($template, '|') === false) {
			$key                = 'default|' . $template;
			$result['template'] = isset($this['form']->getComponent('template')->getItems()[$key]) ? $key : 'default' . '|' . $template;
		} else {
			$result['file'] = explode('|', $template)[1] ?? $template;
		}

		return $result;
	}
}
