<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\Repository\Features;
use DynamicModule\Model\CacheService;
use Nette\Caching\Cache;
use Nette\Utils\Json;
use Nette\Utils\Strings;

class FilterUrlHelper
{
	public static string $keyNav    = 'nav';
	public static string $keyValues = 'values';

	protected EntityManagerDecorator $em;
	protected Features               $features;
	protected CacheService           $cacheService;

	public static array $checkExist = [];

	public function __construct(
		EntityManagerDecorator $em,
		Features               $features,
		CacheService           $cacheService
	)
	{
		$this->em           = $em;
		$this->features     = $features;
		$this->cacheService = $cacheService;
	}

	public function validateRelationData(array $data): ?array
	{
		if (!isset($data[self::$keyNav])) {
			return null;
		}

		$new = [
			self::$keyNav => (int) $data[self::$keyNav],
		];

		foreach ([self::$keyValues] as $group) {
			$new[$group] = [];

			if (isset($data[$group])) {
				foreach ($data[$group] as $v) {
					$new[$group][] = (int) $v;
				}
			}

			sort($new[$group]);
		}

		return $new;
	}

	public function canUseBeautyUrl(array $data): bool
	{
		foreach ($data as $k => $v) {
			if (count($v) > 1) {
				return false;
			}
		}

		return true;
	}

	public function createRelationHash(string $siteIdent, array $data): string
	{
		return md5($siteIdent . '/' . Json::encode($data));
	}

	public function getCacheKeyByHash(string $siteIdent, string $locale, array $relations): string
	{
		return 'dmf/' . $siteIdent . '/' . $locale . '/' . $this->createRelationHash($siteIdent, $relations);
	}

	public function getCacheKeyByUrl(string $siteIdent, string $locale, string $url): string
	{
		return 'dmf/' . $siteIdent . '/' . $locale . '/' . md5($url);
	}

	public function generate(string $title, array $filters, ?Group $group = null): array
	{
		$result = [
			'prefix'     => '',
			'suffix'     => '',
			'text'       => $title,
			'isMultiple' => false,
		];

		$list = [
			'prefix' => [],
			'suffix' => [],
		];

		if (!empty($filters)) {
			$features = $this->features->getFeatures(array_keys($filters));
			$values   = [];

			foreach ($filters as $k => $v) {
				$feature = $features[$k] ?? null;

				if ($feature) {
					if (count($v) > 1) {
						$result['isMultiple'] = true;

						$listKey = 'prefix';
						if ($feature->urlPositionMulti === 1) {
							$listKey = 'suffix';
						}

						$list[$listKey][] = [
							'value'    => $feature->nameMulti,
							'priority' => $feature->urlPositionMultiPriority,
						];
					} else {
						foreach ($v as $vv) {
							$value = $feature->values[$vv] ?? null;

							if ($value) {
								$valueName = $value->name;

								if ($value->nameRod[$group->rod] ?? null) {
									$valueName = $value->nameRod[$group->rod];
								}

								$listKey = 'prefix';
								if ($feature->urlPosition === 1) {
									$listKey = 'suffix';
								}

								$list[$listKey][] = [
									'value'    => $valueName,
									'priority' => $feature->urlPositionPriority,
								];

								$values[$vv] = $features[$k]->values[$vv]->name;
							}
						}
					}
				}
			}

			if (!empty($list['prefix'])) {
				usort($list['prefix'], static fn(array $a, array $b) => $a['priority'] <=> $b['priority']);
			}

			if (!empty($list['suffix'])) {
				usort($list['suffix'], static fn(array $a, array $b) => $a['priority'] <=> $b['priority']);
			}

			if (count($features) !== count($values)) {
				$result['isMultiple'] = true;
			}

			foreach ($list as $k => $v) {
				foreach ($v as $vv) {
					$result[$k] .= ' ' . $vv['value'];
				}
			}
		}

		return $result;
	}

	public function addPrefixSuffix(string $text, string $prefix = '', string $suffix = ''): string
	{
		if ($prefix) {
			$text = Strings::firstLower($text);
		}

		return ucfirst(trim($prefix . ' ' . $this->normalizeText($text) . $suffix));
	}

	public function normalizeText(string $text): string
	{
		$expl = explode(' ', $text);
		$tmp  = substr($expl[0], 1);

		if ($tmp && $tmp === strtolower($tmp)) {
			$text = str_replace($expl[0], strtolower($expl[0]), $text);
		}

		return $text;
	}

	public function findOne(string $siteIdent, string $locale, array $relationData): ?string
	{
		$cacheKey = $this->getCacheKeyByHash($siteIdent, $locale, $relationData);

		return $this->cacheService->filterCache->load($cacheKey, function(&$dep) use ($siteIdent, $locale, $relationData) {
			$dep = [
				Cache::EXPIRATION => '30 minutes',
			];

			/** @var ?string $existUrl */
			$existUrl = $this->em->getConnection()->fetchOne("SELECT vut.url 
						FROM dynamicmodule__virtual_url vu
						INNER JOIN dynamicmodule__virtual_url_text vut ON vu.id = vut.id AND vut.locale = :lang
						WHERE vu.site_ident = :siteIdent AND vu.relation_hash = :relationHash", [
				'siteIdent'    => $siteIdent,
				'lang'         => $locale,
				'relationHash' => $this->createRelationHash($siteIdent, $relationData),
			]) ?: null;

			return $existUrl;
		});
	}

	public function checkExist(array $data): void
	{
		$conn             = $this->em->getConnection();
		$siteIdent        = array_values($data)[0]['siteIdent'];
		$lang             = array_values($data)[0]['lang'];
		$nav              = array_values($data)[0]['nav'];
		$existWithoutText = [];
		$existWithText    = [];

		foreach ($conn->fetchAllAssociative("SELECT vu.id, vu.relation_hash, vut.locale 
					FROM dynamicmodule__virtual_url vu
					LEFT JOIN dynamicmodule__virtual_url_text vut ON vu.id = vut.id AND vut.locale = :lang
					WHERE vu.site_ident = :siteIdent 
					  AND vu.relation_hash IN ('" . implode("','", array_keys($data)) . "')
					  AND vu.navigation = :nav", [
			'lang'      => $lang,
			'siteIdent' => $siteIdent,
			'nav'       => $nav,
		]) as $row) {
			if ($row['locale']) {
				$existWithText[$row['relation_hash']] = $row['id'];
			} else {
				$existWithoutText[$row['relation_hash']] = $row['id'];
			}
		}

		// nove bez vsech dat
		foreach (array_diff_key($data, $existWithText, $existWithoutText) as $hash => $row) {
			$conn->insert('dynamicmodule__virtual_url', [
				'site_ident'    => $siteIdent,
				'relation_hash' => $hash,
				'navigation'    => $nav,
			]);

			$lastId = $conn->lastInsertId();

			foreach ($row['data'][self::$keyValues] as $k => $v) {
				$conn->insert('dynamicmodule__virtual_url_feature_value', [
					'virtual_url_id'   => $lastId,
					'feature_value_id' => $v,
				]);
			}

			$conn->insert('dynamicmodule__virtual_url_text', [
				'id'     => $lastId,
				'locale' => $lang,
				'url'    => $row['url'],
			]);
		}

		// nove bez textu
		foreach ($existWithoutText as $hash => $id) {
			$conn->insert('dynamicmodule__virtual_url_text', [
				'id'     => $id,
				'locale' => $lang,
				'url'    => $data[$hash]['url'],
			]);
		}
	}
}
