<?php declare(strict_types=1);

namespace DynamicModule\FrontModule\Model;

use Closure;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\Model\DynamicModuleConfig;

class GroupQuery
{
	/** @var Closure[] */
	protected array $filter = [];
	/** @var Closure[] */
	protected array $select = [];
	protected string $moduleKey;
	protected string $lang;

	public function __construct(string $moduleKey)
	{
		$this->moduleKey = $moduleKey;
	}

	/**
	 * @param int[] $ids
	 */
	public function filterIds(array $ids): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($ids) {
			if (count($ids) > 0) {
				$qb->andWhere('node.id IN (:ids)')->setParameter('ids', $ids);
			}
		};

		return $this;
	}

	protected function doCreateQuery(EntityRepository $repository): QueryBuilder
	{
		$qb = $this->createBasicDql($repository);

		foreach ($this->select as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	protected function createBasicDql(EntityRepository $repository): QueryBuilder
	{
		$qb = $repository->createQueryBuilder('node')
						 ->select('node, IDENTITY(node.parent) as parent')
						 ->andWhere('node.moduleKey = :module')
						 ->orderBy('node.root, node.lft', 'ASC')
						 ->setParameter('module', $this->moduleKey);

		if (!DynamicModuleConfig::load('multiLangPublication')) {
			$qb->andWhere('node.isPublished = 1');
		}

		foreach ($this->filter as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	public function getQueryBuilder(EntityRepository $repository): QueryBuilder
	{
		return $this->doCreateQuery($repository);
	}

}