<?php declare(strict_types = 1);

namespace DynamicModule\FrontModule\Presenters;

use Core\Components\Navigation\DaoNavigationItem;
use DynamicModule\FrontModule\Components\Filter;
use DynamicModule\FrontModule\Components\IFilterFactory;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\FilterUrlHelper;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\FrontModule\Model\Repository\Members;
use DynamicModule\Model\DynamicModuleConfig;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Helpers\TemplatePagesHelper;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	protected TemplatePages       $templatePagesService;
	protected Paths               $pathsService;
	protected TemplatePagesHelper $pagesHelper;
	protected IGroupsFactory      $groupsFrontRepoFactory;
	protected Members             $members;
	protected FilterUrlHelper     $filterUrlHelper;

	protected array $groups = [];

	public function __construct(
		TemplatePages       $templatePages,
		Paths               $paths,
		TemplatePagesHelper $templatePagesHelper,
		IGroupsFactory      $groupsFrontRepoFactory,
		Members             $members,
		FilterUrlHelper     $filterUrlHelper
	)
	{
		$this->templatePagesService   = $templatePages;
		$this->pathsService           = $paths;
		$this->pagesHelper            = $templatePagesHelper;
		$this->groupsFrontRepoFactory = $groupsFrontRepoFactory;
		$this->members                = $members;
		$this->filterUrlHelper        = $filterUrlHelper;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionGroups(): void
	{
		$activeNav      = $this->getActiveNavigation();
		$templatePageId = (int) $activeNav->componentParams['templatePage'];
		$page           = $templatePageId ? $this->templatePagesService->get($templatePageId) : null;

		if (!$page) {
			$this->error('pages.page.notFound');
		}

		$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-groups";

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds([]);
		$this->template->groups = $groups;

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		$this->template->templateTextValues = $texts;
		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
	}

	public function actionGroup(string $id): void
	{
		$this->autoCanonicalize = false;
		$activeNav              = $this->getActiveNavigation();
		$templatePageId         = (int) ($activeNav->componentParams['templatePageDetail'] ?: $activeNav->componentParams['templatePage']);
		$page                   = $this->templatePagesService->get($templatePageId);

		if (!$page) {
			$this->error('pages.page.notFound');
		}

		$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-{$id}";

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds($activeNav->componentParams['baseGroup'] ?? []);
		$this->template->groups = $groups;

		/** @var ?Group $group */
		$group = $groups ? array_values($groups)[0] : null;

		$this->groups          = $groups;
		$this->template->group = $group;

		$texts = $page->getTexts();
		$texts = $this->pagesHelper->checkTexts($texts);

		$activeNavigation = $this->getActiveNavigation();
		if ($activeNavigation && $activeNavigation->getParam('virtualUrlData')) {
			$virtualUrlData = $activeNavigation->getParam('virtualUrlData');

			if ($virtualUrlData['seo_title']) {
				$this->setPageTitle($virtualUrlData['seo_title']);
			}

			if ($virtualUrlData['seo_description']) {
				$this['meta']->setMeta('description', $virtualUrlData['seo_description']);
			}

			$this['meta']->setMeta('canonical', $this->getHttpRequest()->getUrl()->getBaseUrl() . ltrim($virtualUrlData['url'], '/'));
		} else {
			if (DynamicModuleConfig::load('allowVirtualUrls')) {
				/** @var Filter|null $filterComponent */
				$filterComponent = $this->getComponent('filter', false);

				if ($filterComponent) {
					$filter = array_map(static function($v) { return explode('|', $v); }, $filterComponent->filter);
					if ($filter) {
						$tmp    = $this->filterUrlHelper->generate($group->title, $filter, $group);

						$tmpTitle = $this->filterUrlHelper->addPrefixSuffix($group->title, $tmp['prefix'], $tmp['suffix']);
						$this->setPageTitle($tmpTitle);
						$group->title = $tmpTitle;

						$activeNavigation->setParam('virtualUrlData', [
							'title'     => $tmpTitle,
							'seo_title' => $tmpTitle,
						]);
					}
				}
			}

			$this['meta']->setMeta('canonical', $this->getHttpRequest()->getUrl()->getBaseUrl() . ltrim($this->link('this', ['id' => $id]), '/'));
		}

		$this->template->templateTextValues = $texts;
		$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
	}

	public function actionMember(int $id): void
	{
		$activeNav = $this->getActiveNavigation();
		if (!$activeNav->componentParams['allowMemberDetail']) {
			$this->error('pages.page.notFound');
		}

		$groupsRepo             = $this->groupsFrontRepoFactory->create($activeNav->componentParams['module']);
		$groups                 = $groupsRepo->getByIds($activeNav->componentParams['baseGroup'] ?? []);
		$this->template->groups = $groups;
		$member                 = null;

		foreach ($groups as $group) {
			$member = $group->findMember($id);
			if ($member) {
				$this->template->member = $member;
				$this->template->group  = $group;

				$groupLink = $this->link('group', ['id' => $group->getId()]);
				if ($groupLink && $groupLink !== '#' && $groupLink !== $this->activeNavigation->link) {
					$this->breadcrumbsAfter[] = new DaoNavigationItem([
						'title' => $group->getFieldValue('title') ?: $group->getTitle(),
						'link'  => $groupLink,
					]);
				}

				$this->template->templateTextValues = $member->getKeyValueData();
				break;
			}
		}

		if (!$member) {
			$this->error('pages.page.notFound');
		}

		if ($member->seo && DynamicModuleConfig::load('member.allowSeo')) {
			$this->seoDataFromEntity = (array) $member->seo;
		}

		if (is_null($this->seoDataFromEntity['title']) && DynamicModuleConfig::load('module.' . $member->module . '.member.useFieldAsTitle')) {
			$this->setPageTitle($member->getFieldValue(DynamicModuleConfig::load('module.' . $member->module . '.member.useFieldAsTitle')));
		}

		$this->breadcrumbsAfter[] = new DaoNavigationItem([
			'title' => $member->getFieldValue('title') ?: $member->title,
			'link'  => $this->getHttpRequest()->getUrl()->getAbsoluteUrl(),
		]);

		$templatesDir = sprintf('%s/Front/default/%s/TemplateMember', TEMPLATES_DIR, ucfirst($activeNav->componentParams['module']));

		if ($activeNav->componentParams['templatePageMemberDetail']) {
			$page = $this->templatePagesService->get((int) $activeNav->componentParams['templatePageMemberDetail']);

			if (!$page) {
				$this->error('pages.page.notFound');
			}

			$this->template->pageClass .= " template-page tp-{$page->getId()} dynamic-module dm-{$id}";

			$texts = $page->getTexts();
			$texts = $this->pagesHelper->checkTexts($texts);

			$this->template->templateTextValues = $texts;
			$this->template->setFile($this->pathsService->getTemplatePagesFile($page->getTemplate()));
		} else {
			$this->template->setFile($templatesDir . DS . $this->template->member->getTemplate());
		}
	}

	protected function createComponentFilter(IFilterFactory $factory): Filter
	{
		$control = $factory->create();

		$members = [];
		$loop    = static function(Group $group) use (&$loop, &$members) {
			foreach ($group->members as $member) {
				$members[$member->getId()] = $member;
			}

			foreach ($group->getChildren() as $child) {
				$loop($child);
			}
		};

		foreach ($this->groups as $group) {
			$loop($group);
		}

		if ($this->isAjax()) {
			$this->redrawControl('filter');
			$this->redrawControl('list');
			$this->redrawControl('head');
			$this->redrawControl('description');

			foreach (DynamicModuleConfig::load('filtersRedrawControls') ?: [] as $v) {
				$this->redrawControl($v);
			}
		}

		$control->onFilter[] = function(): void {
			$this->redrawControl('filter');
			$this->redrawControl('list');
			$this->redrawControl('head');
			$this->redrawControl('description');

			foreach (DynamicModuleConfig::load('filtersRedrawControls') ?: [] as $v) {
				$this->redrawControl($v);
			}
		};

		$control->activeNav = $this->getActiveNavigation();
		$control->groups    = $this->groups;
		$control->members   = $members;
		$control->title     = $this->getActiveNavigation()->title;

		return $control;
	}
}
