<?php declare(strict_types = 1);

namespace DynamicModule\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\TemplateReader\Entity\ITemplate;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use DynamicModule\Model\Traits\TFields;
use DynamicModule\Model\Traits\TPublished;
use DynamicModule\Model\Traits\TTemplate;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="dynamicmodule__group")
 * @ORM\Entity(repositoryClass="Gedmo\Tree\Entity\Repository\NestedTreeRepository")
 */
class Group implements ITemplate
{
	use TId;
	use TTemplate;

	/** @deprecated */
	use TPublished;

	use TFields;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $moduleKey;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $title;

	/**
	 * @var Collection<int, GroupMember>
	 * @ORM\OneToMany(targetEntity="GroupMember", mappedBy="group", cascade={"all"})
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	public Collection $members;

	/**
	 * @var Collection<int, Field>
	 * @ORM\ManyToMany(targetEntity="DynamicModule\Model\Entities\Field", cascade={"persist"}, orphanRemoval=true)
	 * @ORM\JoinTable(name="dynamicmodule__group_fields")
	 */
	public Collection $fields;

	/**
	 * @var Collection<int, Group>
	 * @ORM\OneToMany(targetEntity="Group", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	public Collection $children;

	/**
	 * @var Group|null
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Group", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE", nullable=true)
	 */
	public ?Group $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	public int $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	public int $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	public int $rgt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Group")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private Group $root;

	/**
	 * @var Collection<string, GroupText>
	 * @ORM\OneToMany(targetEntity="GroupText", mappedBy="group", indexBy="lang")
	 */
	public Collection $texts;

	/**
	 * @var Collection<GroupFeature>
	 * @ORM\OneToMany(targetEntity="GroupFeature", mappedBy="group", indexBy="feature_id")
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	public Collection $filters;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $rod = null;

	/**
	 * @ORM\Column(type="json", nullable=true)
	 */
	public ?array $params = null;

	public function __construct(string $moduleKey, string $title)
	{
		$this->title     = $title;
		$this->moduleKey = $moduleKey;
		$this->members   = new ArrayCollection;
		$this->children  = new ArrayCollection;
		$this->fields    = new ArrayCollection;
		$this->texts     = new ArrayCollection;
		$this->filters   = new ArrayCollection;
	}

	/**
	 * @return GroupMember[]
	 */
	public function getMembers(): array
	{
		return $this->members->toArray();
	}

	public function addMember(Member $member): void
	{
		$this->members->add(new GroupMember($this, $member));
	}

	public function getRoot(): ?Group
	{
		return $this->root;
	}

	public function setParent(?Group $parent = null): void
	{
		$this->parent = $parent;
	}

	public function getParent(): ?Group
	{
		return $this->parent;
	}

	/**
	 * @return Group[]
	 */
	public function getChildren(): array
	{
		return $this->children->toArray();
	}

	public function getLevel(): int
	{
		return $this->lvl;
	}

	public function addEntityText(GroupText $text): self
	{
		$this->texts->add($text);

		return $this;
	}

	public function setEntityText(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	/**
	 * @return GroupText[]
	 */
	public function getEntityTexts(): array
	{
		$result = [];
		/** @var GroupText $text */
		foreach ($this->texts->toArray() as $text) {
			$result[$text->getLang()] = $text;
		}

		return $result;
	}

}
