<?php declare(strict_types = 1);

namespace DynamicModule\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TLang;
use Core\Model\TemplateReader\Entity\ITemplate;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use DynamicModule\Model\Traits\TFields;
use DynamicModule\Model\Traits\TPublished;
use DynamicModule\Model\Traits\TTemplate;

/**
 * @ORM\Table(name="dynamicmodule__member")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\SortableRepository")
 */
class Member implements ITemplate
{
	use TId;
	use TLang;

	/** @deprecated */
	use TPublished;

	use TTemplate;
	use TFields;

	/**
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public string $title;

	/**
	 * @var Collection<int, GroupMember> $groups
	 * @ORM\OneToMany(targetEntity="GroupMember", mappedBy="member", cascade={"all"}, orphanRemoval=true)
	 */
	public Collection $groups;

	/**
	 * @var Collection<int, Field> $fields
	 * @ORM\ManyToMany(targetEntity="DynamicModule\Model\Entities\Field", cascade={"persist"}, orphanRemoval=true)
	 * @ORM\JoinTable(name="dynamicmodule__member_fields")
	 */
	public Collection $fields;

	/**
	 * @var Collection<string, MemberText>
	 * @ORM\OneToMany(targetEntity="MemberText", mappedBy="member", indexBy="lang")
	 */
	public Collection $texts;

	/**
	 * @var Collection<FeatureValue>
	 * @ORM\ManyToMany(targetEntity="FeatureValue", indexBy="id")
	 * @ORM\JoinTable(name="dynamicmodule__member_feature")
	 */
	public Collection $features;

	/**
	 * @ORM\Column(type="json", nullable=true)
	 */
	public ?array $params = null;

	public function __construct(string $title, string $template)
	{
		$this->title    = $title;
		$this->groups   = new ArrayCollection;
		$this->template = $template;
		$this->fields   = new ArrayCollection;
		$this->texts    = new ArrayCollection;
		$this->features = new ArrayCollection;
	}

	/**
	 * @return Group[]
	 */
	public function getGroups(): array
	{
		return array_map(static fn(GroupMember $groupMember): Group => $groupMember->group, $this->groups->toArray());
	}

	public function addGroup(Group $group): void
	{
		$this->groups->add(new GroupMember($group, $this));
	}

	public function removeGroup(Group $group): void
	{
		/** @var GroupMember $groupMember */
		foreach ($this->groups->toArray() as $groupMember) {
			if ($groupMember->group->getId() === $group->getId() && $groupMember->member->getId() === $this->getId()) {
				$this->groups->removeElement($groupMember);
			}
		}
	}

	public function addEntityText(MemberText $text): self
	{
		$this->texts->add($text);

		return $this;
	}

	public function setEntityText(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	/**
	 * @return MemberText[]
	 */
	public function getEntityTexts(): array
	{
		$result = [];
		/** @var MemberText $text */
		foreach ($this->texts->toArray() as $text) {
			$result[$text->getLang()] = $text;
		}

		return $result;
	}

}
