<?php declare(strict_types = 1);

namespace DynamicModule\Model\Repository;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Lang\Langs;
use Core\Model\TemplateReader\TemplateReader;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use DynamicModule\AdminModule\Model\Events\MemberEvent;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Entities\GroupMember;
use DynamicModule\Model\Entities\Member;
use DynamicModule\Model\Entities\MemberText;
use Exception;
use Gallery\AdminModule\Model\Albums;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Throwable;
use Tracy\Debugger;


/**
 * @method Member[] getAll()
 * @method Member|null get(int $id)
 */
class Members extends BaseEntityService
{
	use TPublish;

	public const CACHE_NAMESPACE = 'dynamicModuleMembers';

	/** @var string */
	protected $entityClass = Member::class;
	/** @var string[] */
	protected array           $checkRoute = [];
	protected Translator      $translator;
	protected Langs           $langs;
	protected Albums          $albums;
	protected Container       $container;
	protected EventDispatcher $eventDispatcher;

	public function __construct(
		Translator      $translator,
		Langs           $langs,
		Albums          $albums,
		Container       $container,
		EventDispatcher $eventDispatcher
	)
	{
		$this->translator      = $translator;
		$this->langs           = $langs;
		$this->albums          = $albums;
		$this->container       = $container;
		$this->eventDispatcher = $eventDispatcher;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getQueryBuilderByModule(string $moduleKey): QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('m');
		$qb->join('m.groups', 'gm', Join::WITH, 'm.id = gm.member')
			->addSelect('gm')
			->join('gm.group', 'g')
			->andWhere('g.moduleKey = :module')
			->setParameter('module', $moduleKey);

		return $qb;
	}

	public function getQueryBuilderByGroup(int $groupId): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('gm')
			->from(GroupMember::class, 'gm')
			->join('gm.group', 'g')
			->where($qb->expr()->eq('g.id', $groupId))
			->addOrderBy('gm.position');

		return $qb;
	}

	/**
	 * @param string|int|null $key
	 *
	 * @throws Throwable
	 */
	public function checkRoute($key = null, ?array $groups, string $module, array $moreFields = []): ?array
	{
		if (!is_array($groups)) {
			$groups = [];
		}

		$mKey = $module . serialize($groups) . (!empty($moreFields) ? ':' . serialize($moreFields) : '') . '_' . $this->translator->getLocale();
		if (!isset($this->checkRoute[$mKey])) {
			$this->checkRoute[$mKey] = $this->getCache()->load($mKey, function(&$dep) use ($groups, $module, $moreFields) {
				$dep = [
					Cache::TAGS       => [self::CACHE_NAMESPACE, self::CACHE_NAMESPACE . 'checkRoute' . $module],
					Cache::EXPIRATION => '1 week',
				];

				$checkRoute = [];

				$rows = $this->getEr()->createQueryBuilder('m')
					->select('m.id, m.title');

				if (!empty($groups)) {
					$rows->innerJoin('m.groups', 'gm', Join::WITH, 'gm.group IN (:groups)')
						->setParameter('groups', $groups);
				} else {
					$rows->innerJoin('m.groups', 'gm');
				}

				$rows->innerJoin('gm.group', 'g', Join::WITH, 'g.moduleKey = :module')
					->setParameter('module', $module);

				if (DynamicModuleConfig::load('multiLangPublication')) {
					$rows->innerJoin('m.texts', 'txt', Join::WITH, 'txt.lang = :lang AND txt.isPublished = 1');
					$rows->setParameter('lang', $this->translator->getLocale());
				}

				foreach ($rows->getQuery()->getArrayResult() as $row) {
					$checkRoute[$row['id']] = [
						'id'    => $row['id'],
						'title' => $row['title'],
					];
				}

				if (!empty($checkRoute)) {
					$moreFields = array_merge($moreFields, ['title', 'alias', 'fullUrl', 'gallery']);

					foreach ($this->em->getConnection()->fetchAllAssociative("SELECT mf.member_id, f.key, f.value FROM dynamicmodule__field f
    				INNER JOIN dynamicmodule__member_fields mf ON mf.field_id = f.id AND mf.member_id IN (" . implode(',', array_keys($checkRoute)) . ")
    				WHERE f.key IN ('" . implode('\',\'', $moreFields) . "') AND (f.lang = '{$this->translator->getLocale()}' OR f.lang IS NULL)") as $row) {
						$checkRoute[$row['member_id']][$row['key']] = $row['value'];
					}

					foreach ($checkRoute as $id => $vals) {
						if (isset($vals['fullUrl']) && $vals['fullUrl'] !== '') {
							$url = $vals['fullUrl'];
						} else {
							$url = $id . '-' . (isset($vals['alias']) && $vals['alias'] !== '' ? $vals['alias'] : Strings::webalize($vals['title']));
						}

						$tmp = [];
						foreach (explode('/', $url) as $part) {
							$tmp[] = Strings::webalize($part);
						}

						$url = implode('/', $tmp);

						$checkRoute[$id]['url'] = $url;
						$checkRoute[$url]       = $checkRoute[$id];
					}
				}

				return $checkRoute;
			});
		}

		return $key ? ($this->checkRoute[$mKey][$key] ?? null) : $this->checkRoute[$mKey];
	}

	public function cloneMember(int $id): bool
	{
		try {
			$member = $this->get($id);
			if (!$member || !$member->getGroups()) {
				return false;
			}

			/** @var Group $anyGroup */
			$anyGroup = Arrays::first($member->getGroups());
			$module   = $anyGroup->moduleKey;

			/** @var TemplateReader $templateReader */
			$templateReader = $this->container->getByType(TemplateReader::class);
			$templateReader->setTemplatesDir(
				sprintf('%s/Front/default/%s/TemplateMember/', TEMPLATES_DIR, ucfirst($module)),
			);
			$templateInputs = $templateReader->getTemplateInputs($member->template);

			$newMember           = clone $member;
			$newMember->groups   = new ArrayCollection;
			$newMember->features = new ArrayCollection;
			$newMember->texts    = new ArrayCollection;
			$newMember->fields   = new ArrayCollection;

			$groupIds = [];
			foreach ($member->getGroups() as $group) {
				$groupIds[] = $group->getId();
				$newMember->addGroup($group);
			}

			foreach ($member->features as $feature) {
				$newMember->features->add($feature);
			}

			foreach ($member->getEntityTexts() as $text) {
				$newText         = clone $text;
				$newText->member = $newMember;

				$newMember->texts->add($newText);
				$this->em->persist($newText);
			}

			foreach ($member->fields as $field) {
				$newField = clone $field;

				foreach ($templateInputs as $input) {
					if ($input['type'] === 'gallery' && ($album = $this->albums->get((int) $newField->value))) {
						$newAlbum        = $this->albums->cloneAlbum($album);
						$newField->value = (string) $newAlbum->getId();
					}
				}

				$newMember->fields->add($newField);
			}

			$this->em->persist($newMember);
			$this->em->flush();

			$this->eventDispatcher->dispatch(
				new MemberEvent($newMember),
				'dynamicModule.admin.' . $newMember->groups->first()->group->moduleKey . '.memberChanged',
			);

			$cache       = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
			$memberCache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

			foreach ($this->langs->getLangs(false) as $lang) {
				foreach ($groupIds as $groupId) {
					$cache->remove($module . '_' . md5((string) $groupId) . '_' . $lang->getTag());
				}

				$memberCache->remove('member/' . $lang->getTag() . '/' . $newMember->getId());
			}
			$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

			$cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
			$cache->clean([Cache::Tags => self::CACHE_NAMESPACE]);

			return true;
		} catch (Exception $ex) {
			Debugger::log($ex->getMessage());

			return false;
		}
	}

	/**
	 * @param string|int $id
	 */
	public function remove($id): bool
	{
		$member = $this->get($id);
		if (!$member) {
			return false;
		}

		$groups = array_map(static fn($g) => (string) $g->getId(), $member->getGroups());
		$result = parent::remove($id);

		$cache = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
		foreach ($this->langs->getLangs(false) as $lang) {
			foreach ($groups as $groupId) {
				$cache->remove(md5($groupId) . '_' . $lang->getTag());
			}
		}
		$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

		return $result;
	}

	public function invertPublishText(int $id, string $lang): bool
	{
		$member = $this->get($id);
		if (!$member) {
			return false;
		}

		/** @var MemberText|null $text */
		$text = $member->texts->get($lang);
		if (!$text) {
			return false;
		}

		$text->isPublished = (int) !$text->isPublished;
		$this->em->persist($text);
		$this->em->flush();

		$groups = array_map(static fn($g) => (string) $g->getId(), $member->getGroups());
		$cache  = new Cache($this->cacheStorage, \DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE);
		foreach ($groups as $groupId) {
			$cache->remove(md5($groupId) . '_' . $lang);
		}
		$cache->clean([Cache::TAGS => [\DynamicModule\FrontModule\Model\Repository\Groups::CACHE_NAMESPACE]]);

		return true;
	}
}
